import cv2
import numpy as np
import RPi.GPIO as GPIO
import time

# X-band radar sensing output connected to GPIO 27 of Rpi
HB100_INPUT_PIN = 27

# configure GPIO pins for input into GPIO 27
GPIO.setmode(GPIO.BCM)
GPIO.setup(HB100_INPUT_PIN, GPIO.IN)

# sample global vars for adjustment of sensitivity/measurements
#   - A greater max-pulse-count will count more pulse inputs, leading
#     to more averaged doppler frequency - it will also take longer
#   - A higher motion-sensitivity will reduce false motion readings
MAX_PULSE_COUNT = 10
MOTION_SENSITIVITY = 10


cap = cv2.VideoCapture(0)  # initialize the camera

safe_distance_color = (0, 255, 0)  # Green color
safe_distance_thickness = 2
safe_distance_position = 280  # Adjust the vertical position as needed


def count_frequency(GPIO_pin, max_pulse_count=10, ms_timeout=50):
    """ Monitors the desired GPIO input pin and measures the frequency
        of an incoming signal. For this example it measures the output of
        a HB100 X-Band Radar Doppler sensor, where the frequency represents
        the measured Doppler frequency due to detected motion.
    """
    start_time = time.time()
    pulse_count = 0

    # count time it takes for 10 pulses
    for count in range(max_pulse_count):

        # wait for falling pulse edge - or timeout after 50 ms
        edge_detected = GPIO.wait_for_edge(GPIO_pin, GPIO.FALLING, timeout=ms_timeout)

        # if pulse detected - iterate count
        if edge_detected is not None:
            pulse_count += 1

    # work out duration of counting and subsequent frequency (Hz)
    duration = time.time() - start_time 
    frequency = pulse_count / duration
    speed = frequency / float (31.36)
    

    

    return frequency

while True:
    _, frame = cap.read()  # read a frame from the camera
    frame = cv2.resize(frame, (640, 480))  # resize the frame

    # apply color thresholding to detect white and yellow lanes
    hsv = cv2.cvtColor(frame, cv2.COLOR_BGR2HSV)
    lower_white = np.array([0, 0, 200])
    upper_white = np.array([180, 30, 255])
    mask_white = cv2.inRange(hsv, lower_white, upper_white)
    lower_yellow = np.array([15, 50, 50])
    upper_yellow = np.array([30, 255, 255])
    mask_yellow = cv2.inRange(hsv, lower_yellow, upper_yellow)
    mask = cv2.bitwise_or(mask_white, mask_yellow)

    # apply Gaussian blur to reduce noise
    blur = cv2.GaussianBlur(mask, (5, 5), 0)

    # apply Canny edge detection to detect edges
    edges = cv2.Canny(blur, 50, 150)

    # apply a region of interest mask to focus on the road lanes
    mask = np.zeros_like(edges)
    mask_color = (255, 255, 255)
    vertices = np.array([[(0, 480), (320, 240), (320, 240), (640, 480)]], dtype=np.int32)
    cv2.fillPoly(mask, vertices, mask_color)
    masked_edges = cv2.bitwise_and(edges, mask)

    # apply Hough line detection to detect lane lines
    rho = 1
    theta = np.pi / 180
    threshold = 15
    min_line_length = 30
    max_line_gap = 10
    lines = cv2.HoughLinesP(masked_edges, rho, theta, threshold, np.array([]), min_line_length, max_line_gap)

    # Initialize variables to store lane boundaries
    left_lane_boundary = None
    right_lane_boundary = None

    # Detect and draw the lane lines on the original frame
    if lines is not None:
        for line in lines:
            x1, y1, x2, y2 = line[0]

            # Calculate the slope of the line
            slope = (y2 - y1) / (x2 - x1)

            # Filter out near-horizontal lines (likely not lane lines)
            if abs(slope) > 0.5:
                if slope < 0:
                    # Lines with negative slope are part of the left lane boundary
                    if left_lane_boundary is None or x1 < left_lane_boundary[0]:
                        left_lane_boundary = (x1, y1, x2, y2)
                else:
                    # Lines with positive slope are part of the right lane boundary
                    if right_lane_boundary is None or x1 > right_lane_boundary[2]:
                        right_lane_boundary = (x1, y1, x2, y2)

        # Draw the left lane boundary
        if left_lane_boundary is not None:
            x1, y1, x2, y2 = left_lane_boundary
            cv2.line(frame, (x1, y1), (x2, y2), (0, 0, 255), 3)  # Red color for left lane

        # Draw the right lane boundary
        if right_lane_boundary is not None:
            x1, y1, x2, y2 = right_lane_boundary
            cv2.line(frame, (x1, y1), (x2, y2), (0, 0, 255), 3)  # Red color for right lane

        # Draw the safe distance line between lane boundaries
        if left_lane_boundary is not None and right_lane_boundary is not None:
            safe_distance_position = 280  
            cv2.line(frame, (left_lane_boundary[0], safe_distance_position),
                     (right_lane_boundary[2], safe_distance_position),
                     safe_distance_color, safe_distance_thickness)
    doppler_freq = count_frequency(HB100_INPUT_PIN)
    speed = doppler_freq / float (31.36)
    print (speed)
    font = cv2.FONT_HERSHEY_DUPLEX
    cv2.putText(frame,str (speed), (250, 250), font, 1.0, (40, 155, 155), 2,cv2.LINE_4)

            

    # Display the resulting frame
    cv2.imshow('frame', frame)

    # Exit the loop when 'q' is pressed
    if cv2.waitKey(1) & 0xFF == ord('q'):
        break

# Release the camera and close all windows
cap.release()
cv2.destroyAllWindows()

