#include <LSM303AGR_ACC_Sensor.h>
#include <LSM303AGR_MAG_Sensor.h>
#include <WiFi.h>
#include <WebServer.h>

const char* ssid = "JioFiber-4G";
const char* password = "1234567890";

#define DECLINATION_ANGLE_DEGREES (-0.96f)  // This is the declination angle at your location (find it at https://www.magnetic-declination.com/)
#define PI 3.14159265359

#define MOTION_THRESHOLD 1000 // Adjust this threshold based on your accelerometer's sensitivity

#define HTTP_PORT 80

WebServer server(HTTP_PORT);

// Components.
LSM303AGR_ACC_Sensor Acc(&Wire);
LSM303AGR_MAG_Sensor Mag(&Wire);

void setup() {
  Serial.begin(115200);
  
  // Connect to Wi-Fi
  connectToWiFi();

  // Initialize I2C bus.
  Wire.begin();

  // Initialize components.
  Acc.begin();
  Acc.Enable();
  Mag.begin();
  Mag.Enable();

  // Route for root / web page
  server.on("/", HTTP_GET, handleRoot);

  // Start server
  server.begin();

  Serial.println("HTTP server started");
}

void loop() {
  server.handleClient();
}

void handleRoot() {
  int32_t accelerometer[3];
  Acc.GetAxes(accelerometer);
  
  // Read magnetometer LSM303AGR.
  int32_t magnetometer[3];
  Mag.GetAxes(magnetometer);

  // Calculate heading angle
  float heading = atan2(magnetometer[1], magnetometer[0]);
  heading += DECLINATION_ANGLE_DEGREES * PI / 180;
  if (heading < 0) heading += 2 * PI;
  if (heading > 2 * PI) heading -= 2 * PI;
  float headingDegrees = heading * 180 / PI;

  // Get RSSI
  int32_t rssi = WiFi.RSSI();

  // Calculate distance based on empirical formula
  float distance = calculateDistance(rssi);

  // Determine direction
  String direction;
  if (headingDegrees < 22.5 || headingDegrees >= 337.5)
    direction = "N";
  else if (headingDegrees >= 22.5 && headingDegrees < 67.5)
    direction = "NE";
  else if (headingDegrees >= 67.5 && headingDegrees < 112.5)
    direction = "E";
  else if (headingDegrees >= 112.5 && headingDegrees < 157.5)
    direction = "SE";
  else if (headingDegrees >= 157.5 && headingDegrees < 202.5)
    direction = "S";
  else if (headingDegrees >= 202.5 && headingDegrees < 247.5)
    direction = "SW";
  else if (headingDegrees >= 247.5 && headingDegrees < 292.5)
    direction = "W";
  else if (headingDegrees >= 292.5 && headingDegrees < 337.5)
    direction = "NW";

  // Check for motion based on acceleration
  bool isMoving = isMotionDetected(accelerometer);

  String webpage = "<!DOCTYPE html><html><head><meta name=\"viewport\" content=\"width=device-width, initial-scale=1\"><style>";
  webpage += ".card { box-shadow: 0 4px 8px 0 rgba(0, 0, 0, 0.2); transition: 0.3s; width: 300px; margin: 15px; padding: 10px; border-radius: 10px; }";
  webpage += ".container { padding: 2px 16px; }";
  webpage += "</style></head><body>";
  webpage += "<script>setTimeout(function() { location.reload(); }, 5000);</script>"; // Auto-refresh every 5 seconds
  
  webpage += "<h2 style=\"text-align: center;\">Sensor Data</h2>";

  webpage += "<div class=\"card\"><div class=\"container\">";
  webpage += "<h4>Heading</h4>";
  webpage += "<p>" + String(headingDegrees) + "°</p></div></div>";

  webpage += "<div class=\"card\"><div class=\"container\">";
  webpage += "<h4>Direction</h4>";
  webpage += "<p>" + direction + "</p></div></div>";

  webpage += "<div class=\"card\"><div class=\"container\">";
  webpage += "<h4>Motion</h4>";
  webpage += "<p>" + String(isMoving ? "Yes" : "No") + "</p></div></div>";

  webpage += "<div class=\"card\"><div class=\"container\">";
  webpage += "<h4>RSSI</h4>";
  webpage += "<p>" + String(rssi) + "</p></div></div>";

  webpage += "<div class=\"card\"><div class=\"container\">";
  webpage += "<h4>Distance</h4>";
  webpage += "<p>" + String(distance) + "m</p></div></div>";

  webpage += "</body></html>";

  server.send(1000, "text/html", webpage);
}

void connectToWiFi() {
  Serial.println("Connecting to WiFi...");
  
  WiFi.begin(ssid, password);

  int attempts = 0;
  while (WiFi.status() != WL_CONNECTED) {
    delay(1000);
    Serial.println("Connecting to WiFi...");
    attempts++;
    if (attempts > 10) {
      Serial.println("Connection failed. Retrying...");
      WiFi.begin(ssid, password);
      attempts = 0;
    }
  }
  
  Serial.println("Connected to WiFi. IP address: " + WiFi.localIP().toString());
}

float calculateDistance(int32_t rssi) {
  // Empirical formula to calculate distance based on RSSI
  // You need to calibrate this formula based on your environment and hardware
  // This is just an example and might not be accurate for all scenarios
  int32_t txPower = -65; // This is the signal strength in dBm at 1 meter distance from the access point
  float exponent = (txPower - rssi) / (10 * 2.0); // 2.0 is an attenuation factor
  float distance = pow(10, exponent);
  return distance;
}

bool isMotionDetected(int32_t accelerometer[3]) {
  // Calculate the magnitude of acceleration
  float accelerationMagnitude = sqrt(accelerometer[0] * accelerometer[0] +
                                     accelerometer[1] * accelerometer[1] +
                                     accelerometer[2] * accelerometer[2]);
  
  // Check if acceleration magnitude exceeds threshold
  return accelerationMagnitude > MOTION_THRESHOLD;
}
