#include <WiFi.h>
#include <AsyncTCP.h>
#include <ESPAsyncWebServer.h>
#include <LDS_YDLIDAR_X2_X2L.h>

// Replace with your network credentials
const char* ssid = "ESP32-LiDAR-AP";
const char* password = "12345678";

// Create AsyncWebServer object on port 80
AsyncWebServer server(80);

LDS_YDLIDAR_X2_X2L lidar;

// Variables to hold the angle and distance data
float angle = 0.0;
float distance = 0.0;

// HTML content
const char htmlPage[] PROGMEM = R"rawliteral(
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>ESP32-S2 LiDAR Map</title>
  <style>
    body { font-family: Arial, sans-serif; text-align: center; margin-top: 50px; }
    .card { display: inline-block; margin: 20px; padding: 20px; border: 1px solid #ccc; border-radius: 10px; }
    canvas { margin-top: 20px; }
  </style>
</head>
<body>
  <h1>Real-Time LiDAR Map</h1>
  <div class="card">
    <h2>Angle</h2>
    <p id="angle">0°</p>
  </div>
  <div class="card">
    <h2>Distance</h2>
    <p id="distance">0 mm</p>
  </div>
  <canvas id="lidarMap" width="400" height="400"></canvas>
  <script>
    const ctx = document.getElementById('lidarMap').getContext('2d');

    function drawMap(angle, distance) {
      ctx.clearRect(0, 0, ctx.canvas.width, ctx.canvas.height);
      ctx.beginPath();
      const radian = angle * (Math.PI / 180);
      const x = (ctx.canvas.width / 2) + (distance * Math.cos(radian)) / 10;
      const y = (ctx.canvas.height / 2) + (distance * Math.sin(radian)) / 10;
      ctx.moveTo(ctx.canvas.width / 2, ctx.canvas.height / 2);
      ctx.lineTo(x, y);
      ctx.stroke();
    }

    function updateData() {
      fetch('/data')
        .then(response => response.json())
        .then(data => {
          document.getElementById('angle').textContent = data.angle + '°';
          document.getElementById('distance').textContent = data.distance + ' mm';
          drawMap(data.angle, data.distance);
        })
        .catch(error => console.error('Error:', error));
    }

    setInterval(updateData, 1000); // Update every second
  </script>
</body>
</html>
)rawliteral";

void setup() {
  Serial.begin(115200);

  // Set the ESP32-S2 as an access point
  WiFi.softAP(ssid, password);
  IPAddress IP = WiFi.softAPIP();
  Serial.print("AP IP Address: ");
  Serial.println(IP);

  // Serve the HTML page
  server.on("/", HTTP_GET, [](AsyncWebServerRequest *request){
    request->send_P(200, "text/html", htmlPage);
  });

  // Serve the LiDAR data
  server.on("/data", HTTP_GET, [](AsyncWebServerRequest *request){
    String jsonData = "{\"angle\":" + String(angle) + ",\"distance\":" + String(distance) + "}";
    request->send(200, "application/json", jsonData);
  });

  server.begin();

  // Initialize the LiDAR
  lidar.setSerialWriteCallback(lidar_serial_write_callback);
  lidar.setSerialReadCallback(lidar_serial_read_callback);
  lidar.setScanPointCallback(lidar_scan_point_callback);
  lidar.init();

  LDS::result_t result = lidar.start();
  if (result < 0) {
    Serial.println("LiDAR initialization failed!");
  }
}

int lidar_serial_read_callback() {
  return Serial1.read();
}

size_t lidar_serial_write_callback(const uint8_t *buffer, size_t length) {
  return Serial1.write(buffer, length);
}

void lidar_scan_point_callback(float angle_deg, float distance_mm, float quality, bool scan_completed) {
  angle = angle_deg;
  distance = distance_mm;
}

void loop() {
  lidar.loop();
}
