#include "ACS712.h"
#include <U8g2lib.h>
#include <Wire.h>

// ACS712 initialization
ACS712 ACS(A0, 3.3, 4095, 185); // Adjust for ESP32-S2: pin A0, 3.3V reference, 4095 ADC max, 185 mV/A

// Mode definitions
#define MODE_AC_CURRENT 0
#define MODE_DC_CURRENT 1
#define MODE_FREQ 2
#define MODE_AC_ENERGY 3
#define MODE_DC_ENERGY 4

// Touch pin for mode switching
#define TOUCH_PIN 3

// Display initialization
U8G2_SSD1306_128X64_NONAME_F_HW_I2C u8g2(U8G2_R0, /* reset=*/ U8X8_PIN_NONE);

// Variables
int mode = MODE_AC_CURRENT; // Initial mode
float acVoltage = 220.0;    // Default AC voltage
float dcVoltage = 12.0;     // Default DC voltage
float acEnergy = 0.0;       // Total AC energy in kWh
float dcEnergy = 0.0;       // Total DC energy in kWh
unsigned long lastEnergyUpdate = 0; // Last energy update timestamp
unsigned long energyUpdateInterval = 1000; // Energy update interval in ms
unsigned long lastTouchTime = 0; // Debounce for touch
int lastTouchValue = 0; // Variable to store the last touch reading

void setup() {
  Serial.begin(115200);
  delay(1000); // Allow time to open serial monitor
  Serial.println("ACS712 Multi-Mode Energy Monitor");

  // ACS712 auto-calibrate midpoint for current measurement
  ACS.autoMidPoint();

  // Display setup
  u8g2.begin();
  u8g2.setFont(u8g2_font_amstrad_cpc_extended_8u);
  u8g2.clearBuffer();
  u8g2.setCursor(10, 30);
  u8g2.println("Energy Monitor");
  u8g2.sendBuffer();
  delay(1000);
}

void loop() {
  handleTouchInput();
  handleSerialInput();
  updateEnergyUsage();
  updateDisplay(); // Update display data

  switch (mode) {
    case MODE_AC_CURRENT:
      measureACCurrent();
      break;
    case MODE_DC_CURRENT:
      measureDCCurrent();
      break;
    case MODE_FREQ:
      detectFrequency();
      break;
    case MODE_AC_ENERGY:
      displayACEnergy();
      break;
    case MODE_DC_ENERGY:
      displayDCEnergy();
      break;
  }
}

// Handle touch input for mode switching
void handleTouchInput() {
  int currentTouchValue = touchRead(TOUCH_PIN);
  int difference = abs(currentTouchValue - lastTouchValue); // Calculate difference for touch detection
  unsigned long currentTime = millis();

  if (difference > 1000 && (currentTime - lastTouchTime > 500)) { // Debounce touch
    mode = (mode + 1) % 5; // Cycle through modes
    lastTouchTime = currentTime;

    switch (mode) {
      case MODE_AC_CURRENT:
        Serial.println("Mode: AC Current Measurement");
        break;
      case MODE_DC_CURRENT:
        Serial.println("Mode: DC Current Measurement");
        break;
      case MODE_FREQ:
        Serial.println("Mode: Frequency Detection");
        break;
      case MODE_AC_ENERGY:
        Serial.println("Mode: AC Energy Monitoring");
        break;
      case MODE_DC_ENERGY:
        Serial.println("Mode: DC Energy Monitoring");
        break;
    }
  }
  lastTouchValue = currentTouchValue; // Update last touch value
}

// Handle serial input for setting voltages
void handleSerialInput() {
  if (Serial.available() > 0) {
    String input = Serial.readStringUntil('\n');
    input.trim();

    if (input.startsWith("SET_AC")) {
      acVoltage = input.substring(7).toFloat();
      Serial.print("AC Voltage set to: ");
      Serial.println(acVoltage);
    } else if (input.startsWith("SET_DC")) {
      dcVoltage = input.substring(7).toFloat();
      Serial.print("DC Voltage set to: ");
      Serial.println(dcVoltage);
    } else {
      Serial.println("Invalid command.");
    }
  }
}

// Measure AC current
void measureACCurrent() {
  float acCurrent = ACS.mA_AC() / 1000.0; // Convert to A
  Serial.print("AC Current: ");
  Serial.print(acCurrent);
  Serial.println(" A");
}

// Measure DC current
void measureDCCurrent() {
  float dcCurrent = ACS.mA_DC() / 1000.0; // Convert to A
  Serial.print("DC Current: ");
  Serial.print(dcCurrent);
  Serial.println(" A");
}

// Detect frequency
void detectFrequency() {
  float frequency = ACS.detectFrequency(45); // Detect frequency in Hz
  Serial.print("Frequency: ");
  Serial.print(frequency);
  Serial.println(" Hz");
}

// Update and calculate energy usage
void updateEnergyUsage() {
  if (millis() - lastEnergyUpdate >= energyUpdateInterval) {
    float acCurrent = ACS.mA_AC() / 1000.0; // AC current in A
    float dcCurrent = ACS.mA_DC() / 1000.0; // DC current in A

    // Calculate power and update energy
    acEnergy += (acVoltage * acCurrent) / 1000.0 / 3600.0 * (energyUpdateInterval / 1000.0); // kWh
    dcEnergy += (dcVoltage * dcCurrent) / 1000.0 / 3600.0 * (energyUpdateInterval / 1000.0); // kWh

    lastEnergyUpdate = millis();
  }
}

// Update display
void updateDisplay() {
  u8g2.clearBuffer();
  u8g2.clearBuffer();
  u8g2.setFont(u8g2_font_open_iconic_app_4x_t);
  u8g2.drawGlyph(3, 45, 72); 
  u8g2.setFont(u8g2_font_profont17_tr);

  // Display current mode
  u8g2.setCursor(9, 10);
  switch (mode) {
    case MODE_AC_CURRENT:
      u8g2.println("AC Current");
      break;
    case MODE_DC_CURRENT:
      u8g2.println("DC Current");
      break;
    case MODE_FREQ:
      u8g2.println("Frequency");
      break;
    case MODE_AC_ENERGY:
      u8g2.println("Energy");
      break;
    case MODE_DC_ENERGY:
      u8g2.println("DC Energy");
      break;
  }

  u8g2.setCursor(33, 30);
  if (mode == MODE_AC_CURRENT || mode == MODE_DC_CURRENT) {
    u8g2.println((mode == MODE_AC_CURRENT) ? ACS.mA_AC() / 1000.0 : ACS.mA_DC() / 1000.0);
    u8g2.print(" A");
  } else if (mode == MODE_FREQ) {
    u8g2.println(ACS.detectFrequency(45));
    u8g2.print(" Hz");
  } else if (mode == MODE_AC_ENERGY) {
    u8g2.println(acEnergy);
    u8g2.print(" kWh");
  } else if (mode == MODE_DC_ENERGY) {
    u8g2.println(dcEnergy);
    u8g2.print(" kWh");

    // Display DC current below the energy value
    u8g2.setCursor(33, 50);
    float dcCurrent = ACS.mA_DC() / 1000.0; // Calculate DC current
    u8g2.print("I: ");
    u8g2.print(dcCurrent);
    u8g2.print(" A");
  }

  u8g2.sendBuffer();
  delay(100);
}


// Display AC energy
void displayACEnergy() {
  Serial.print("AC Voltage: ");
  Serial.print(acVoltage);
  Serial.println(" V");
  Serial.print("AC Energy: ");
  Serial.print(acEnergy);
  Serial.println(" kWh");
}

// Display DC energy
void displayDCEnergy() {
  Serial.print("DC Voltage: ");
  Serial.print(dcVoltage);
  Serial.println(" V");
  Serial.print("DC Energy: ");
  Serial.print(dcEnergy);
  Serial.println(" kWh");
}
