#include <WiFi.h>
#include <ESPAsyncWebServer.h>

// Wi-Fi credentials for AP mode
const char* ssid = "IndusVoltmeter";
const char* password = "12345678";

// ADC Pin
const int adcPin = 4; // GPIO4

// Voltage Divider for 24V Range
const float R1 = 100000.0; // Resistor R1 value in ohms
const float R2 = 15000.0;  // Resistor R2 value in ohms

// ADC Parameters
const float ADC_MAX_VOLTAGE = 3.3; // Max input voltage to the ADC
const int ADC_RESOLUTION = 4096;  // 12-bit ADC resolution

// Voltage Range Settings
bool is24VRange = false; // Start with 3.3V range

// Create server
AsyncWebServer server(80);

// HTML, CSS, and JS for the Web UI
const char* htmlPage = R"rawliteral(
<!DOCTYPE html>
<html>
<head>
    <title>IndusVoltmeter</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            text-align: center;
            margin: 0;
            padding: 0;
            background-color: #f4f4f9;
        }
        .container {
            margin: 2rem auto;
            max-width: 500px;
            padding: 20px;
            background: #ffffff;
            box-shadow: 0px 4px 8px rgba(0, 0, 0, 0.2);
            border-radius: 10px;
        }
        h1 {
            color: #333;
        }
        .voltage-display {
            font-size: 2rem;
            color: #555;
        }
        .toggle-button {
            display: inline-block;
            margin-top: 1.5rem;
            padding: 10px 20px;
            font-size: 1rem;
            color: #fff;
            background: #0078D4;
            border: none;
            border-radius: 5px;
            cursor: pointer;
        }
        .toggle-button:hover {
            background: #005a9e;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>IndusVoltmeter</h1>
        <p class="voltage-display">Voltage: <span id="voltage">--</span> V</p>
        <button id="rangeToggle" class="toggle-button">Switch to 24V Range</button>
    </div>
    <script>
        const voltageElement = document.getElementById("voltage");
        const toggleButton = document.getElementById("rangeToggle");
        let range = "3.3"; // Start with the 3.3V range

        // Function to fetch voltage data
        async function fetchVoltage() {
            const response = await fetch("/voltage");
            const data = await response.json();
            voltageElement.textContent = data.voltage.toFixed(2);
        }

        // Function to toggle range
        toggleButton.addEventListener("click", async () => {
            range = range === "3.3" ? "24" : "3.3";
            const response = await fetch(`/setrange?range=${range}`);
            const data = await response.json();
            toggleButton.textContent = `Switch to ${data.nextRange}V Range`;
        });

        // Periodically fetch voltage
        setInterval(fetchVoltage, 1000);
    </script>
</body>
</html>
)rawliteral";

void setup() {
  // Initialize serial communication
  Serial.begin(115200);

  // Configure ADC
  analogReadResolution(12); // Set ADC resolution to 12 bits
  analogSetAttenuation(ADC_11db); // Configure ADC for 0-3.3V range

  // Start AP Mode
  WiFi.softAP(ssid, password);
  Serial.println("Access Point Started");
  Serial.print("IP Address: ");
  Serial.println(WiFi.softAPIP());

  // Serve the HTML Page
  server.on("/", HTTP_GET, [](AsyncWebServerRequest* request) {
    request->send_P(200, "text/html", htmlPage);
  });

  // Endpoint to Get Voltage
  server.on("/voltage", HTTP_GET, [](AsyncWebServerRequest* request) {
    float rawADC = analogRead(adcPin);
    float measuredVoltage = (rawADC * ADC_MAX_VOLTAGE / ADC_RESOLUTION);
    float actualVoltage = is24VRange ? (measuredVoltage * ((R1 + R2) / R2)) : measuredVoltage;

    String json = "{\"voltage\": " + String(actualVoltage, 2) + "}";
    request->send(200, "application/json", json);
  });

  // Endpoint to Set Voltage Range
  server.on("/setrange", HTTP_GET, [](AsyncWebServerRequest* request) {
    if (request->hasParam("range")) {
      String range = request->getParam("range")->value();
      is24VRange = (range == "24");
    }

    String nextRange = is24VRange ? "3.3" : "24";
    String json = "{\"nextRange\": \"" + nextRange + "\"}";
    request->send(200, "application/json", json);
  });

  // Start server
  server.begin();
}

void loop() {
  // Main loop can be empty since everything is handled by the server
}
