#include <WiFi.h>
#include <WebServer.h>
#include <Adafruit_NeoPixel.h>

#define LED_PIN  21    // Pin connected to the WS2812B LEDs
#define NUM_LEDS 16   // Number of LEDs in the ring

// Create NeoPixel strip object
Adafruit_NeoPixel strip(NUM_LEDS, LED_PIN, NEO_GRB + NEO_KHZ800);

// Wi-Fi credentials
const char* ssid = "ESP32S2_LED_Controller";
const char* password = "12345678";

// Web server on port 80
WebServer server(80);

// Variables for LED control
uint8_t brightness = 50; // Global brightness (0-255)
uint32_t ledColors[NUM_LEDS] = {0}; // Store color for each LED

// HTML webpage content
const char* webpage = R"rawliteral(
<!DOCTYPE html>
<html lang="en">
<head>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ESP32 LED Controller</title>
    <style>
        body { font-family: Arial, sans-serif; text-align: center; background-color: #f0f0f0; }
        .tab { display: none; }
        .tab.active { display: block; }
        button { margin: 10px; padding: 10px 20px; font-size: 16px; }
        #colorSpectrum { width: 100%; height: 30px; background: linear-gradient(to right, red, yellow, green, cyan, blue, magenta, red); }
        #ring { display: flex; justify-content: center; flex-wrap: wrap; width: 150px; margin: 20px auto; }
        .led { width: 30px; height: 30px; border-radius: 50%; margin: 5px; background: #ccc; cursor: pointer; }
    </style>
</head>
<body>
    <h1>ESP32 LED Controller</h1>
    <nav>
        <button onclick="showTab('control')">RGB Control</button>
        <button onclick="showTab('animations')">Animations</button>
        <button onclick="showTab('ring')">LED Ring</button>
    </nav>

    <!-- Tab 1: RGB Control -->
    <div id="control" class="tab active">
        <h2>RGB Control</h2>
        <div id="colorSpectrum"></div>
        <input type="range" id="red" min="0" max="255" value="255"> R<br>
        <input type="range" id="green" min="0" max="255" value="0"> G<br>
        <input type="range" id="blue" min="0" max="255" value="0"> B<br>
        <input type="number" id="ledIndex" min="0" max="15" placeholder="LED Index (0-15)"><br>
        <input type="color" id="colorPicker">
        <button onclick="updateColor()">Set Color</button>
        <button onclick="toggleLED()">On/Off</button>
    </div>

    <!-- Tab 2: Animations -->
    <div id="animations" class="tab">
        <h2>Custom Animations</h2>
        <button onclick="runAnimation('rainbow')">Rainbow</button>
        <button onclick="runAnimation('snake')">Snake</button>
        <button onclick="runAnimation('breathe')">Breathing</button>
    </div>

    <!-- Tab 3: LED Ring -->
    <div id="ring" class="tab">
        <h2>LED Ring</h2>
        <div id="ring">
            <!-- LED placeholders -->
            <script>
                for (let i = 0; i < 16; i++) {
                    document.write(`<div class='led' id='led${i}' onclick='selectLED(${i})'></div>`);
                }
            </script>
        </div>
    </div>

    <script>
        // Tab switching
        function showTab(id) {
            document.querySelectorAll('.tab').forEach(tab => tab.classList.remove('active'));
            document.getElementById(id).classList.add('active');
        }

        // Update color
        function updateColor() {
            const r = document.getElementById('red').value;
            const g = document.getElementById('green').value;
            const b = document.getElementById('blue').value;
            const index = document.getElementById('ledIndex').value;
            fetch(`/setColor?r=${r}&g=${g}&b=${b}&index=${index}`);
        }

        // Toggle LED
        function toggleLED() {
            const index = document.getElementById('ledIndex').value;
            fetch(`/toggle?index=${index}`);
        }

        // Run animation
        function runAnimation(name) {
            fetch(`/runAnimation?name=${name}`);
        }

        // Select LED
        function selectLED(index) {
            const color = prompt('Enter color (e.g., #ff0000):');
            if (color) fetch(`/selectLED?index=${index}&color=${color}`);
        }
    </script>
</body>
</html>
)rawliteral";

void setup() {
  strip.begin();
  strip.show();
  strip.setBrightness(brightness);

  WiFi.softAP(ssid, password);
  server.on("/", []() { server.send(200, "text/html", webpage); });
  server.on("/setColor", handleSetColor);
  server.on("/toggle", handleToggle);
  server.on("/runAnimation", handleRunAnimation);
  server.on("/selectLED", handleSelectLED);
  server.begin();
}

void loop() {
  server.handleClient();
}

void handleSetColor() {
  int r = server.arg("r").toInt();
  int g = server.arg("g").toInt();
  int b = server.arg("b").toInt();
  int index = server.arg("index").toInt();
  if (index >= 0 && index < NUM_LEDS) {
    strip.setPixelColor(index, strip.Color(r, g, b));
    strip.show();
  }
  server.send(200, "text/plain", "OK");
}

void handleToggle() {
  int index = server.arg("index").toInt();
  if (index >= 0 && index < NUM_LEDS) {
    uint32_t currentColor = strip.getPixelColor(index);
    strip.setPixelColor(index, currentColor == 0 ? 0xFFFFFF : 0);
    strip.show();
  }
  server.send(200, "text/plain", "OK");
}

void handleRunAnimation() {
  String name = server.arg("name");
  if (name == "rainbow") {
    // Add rainbow animation logic
  } else if (name == "snake") {
    // Add snake animation logic
  } else if (name == "breathe") {
    // Add breathing animation logic
  }
  server.send(200, "text/plain", "Running: " + name);
}

void handleSelectLED() {
  int index = server.arg("index").toInt();
  String color = server.arg("color");
  if (index >= 0 && index < NUM_LEDS) {
    long hexColor = strtol(color.substring(1).c_str(), NULL, 16);
    strip.setPixelColor(index, hexColor);
    strip.show();
  }
  server.send(200, "text/plain", "OK");
}
