#include <WiFi.h>
#include <AsyncTCP.h>
#include <ESPAsyncWebServer.h>

#define RADAR_PIN 4  // HB100 IF output connected to GPIO4

// Radar variables
volatile unsigned long pulseCount = 0;
unsigned long lastMillis = 0;
unsigned long pulsePerSecond = 0;
float speed_mps = 0;
float speed_kmph = 0;

// Web Server
AsyncWebServer server(80);
AsyncWebSocket ws("/ws");

// AP Credentials
const char* ssid = "IndusRadar";
const char* password = "12345678";

// Web Page
const char htmlPage[] PROGMEM = R"rawliteral(
<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>IndusRadar Speed Monitor</title>
  <style>
    body {
      background: #000;
      color: #fff;
      text-align: center;
      font-family: Arial, sans-serif;
    }
    h1 { margin: 10px; }
    canvas { background: #111; display: block; margin: auto; border: 1px solid #444; }
    #speedInfo { font-size: 18px; margin-top: 10px; }
  </style>
</head>
<body>
  <h1>IndusRadar Live Speed Monitor</h1>
  <canvas id="radarCanvas" width="600" height="400"></canvas>
  <div id="speedInfo">Speed: 0 m/s | 0 km/h</div>

  <script>
    const canvas = document.getElementById('radarCanvas');
    const ctx = canvas.getContext('2d');
    const centerX = canvas.width / 2;
    const centerY = canvas.height / 2;
    let echoStrength = 0;
    let speed_mps = 0;
    let speed_kmph = 0;

    const ws = new WebSocket('ws://' + location.host + '/ws');
    ws.onmessage = (event) => {
      let data = JSON.parse(event.data);
      speed_mps = data.speed_mps;
      speed_kmph = data.speed_kmph;
      echoStrength = Math.min(speed_kmph / 50, 1);  // normalize
      document.getElementById('speedInfo').innerText = `Speed: ${speed_mps.toFixed(2)} m/s | ${speed_kmph.toFixed(2)} km/h`;
    };

    function drawRadar() {
      ctx.clearRect(0, 0, canvas.width, canvas.height);

      // Transmit waves
      for (let i = 0; i < 5; i++) {
        const radius = (Date.now()/5 + i * 30) % 300;
        ctx.beginPath();
        ctx.arc(centerX, centerY, radius, 0, 2 * Math.PI);
        ctx.strokeStyle = `rgba(255, 0, 0, ${1 - i * 0.2})`;
        ctx.lineWidth = 1.5;
        ctx.stroke();
      }

      // Echo signal (shaded)
      const echoRadius = 80 + echoStrength * 200;
      const gradient = ctx.createRadialGradient(centerX, centerY, 30, centerX, centerY, echoRadius);
      gradient.addColorStop(0, `rgba(0, 200, 255, ${echoStrength})`);
      gradient.addColorStop(1, 'rgba(0,0,0,0)');
      ctx.beginPath();
      ctx.arc(centerX, centerY, echoRadius, 0, 2 * Math.PI);
      ctx.fillStyle = gradient;
      ctx.fill();

      requestAnimationFrame(drawRadar);
    }

    drawRadar();
  </script>
</body>
</html>
)rawliteral";

// Radar ISR
void IRAM_ATTR radarISR() {
  pulseCount++;
}

// WebSocket event handler
void onWebSocketEvent(AsyncWebSocket *server, AsyncWebSocketClient *client,
                      AwsEventType type, void *arg, uint8_t *data, size_t len) {
  // No incoming messages needed
}

void setup() {
  Serial.begin(115200);

  pinMode(RADAR_PIN, INPUT);
  attachInterrupt(digitalPinToInterrupt(RADAR_PIN), radarISR, RISING);

  // Start WiFi AP
  WiFi.mode(WIFI_AP);
  WiFi.softAP(ssid, password);
  Serial.print("AP IP Address: ");
  Serial.println(WiFi.softAPIP());

  // WebSocket setup
  ws.onEvent(onWebSocketEvent);
  server.addHandler(&ws);

  // Serve page
  server.on("/", HTTP_GET, [](AsyncWebServerRequest *request){
    request->send_P(200, "text/html", htmlPage);
  });

  server.begin();
  Serial.println("Web server started");
}

void loop() {
  static unsigned long lastSend = 0;
  unsigned long currentMillis = millis();

  if (currentMillis - lastMillis >= 1000) {
    lastMillis = currentMillis;

    noInterrupts();
    pulsePerSecond = pulseCount;
    pulseCount = 0;
    interrupts();

    speed_mps = pulsePerSecond * 0.02652;
    speed_kmph = speed_mps * 3.6;

    Serial.printf("Pulses: %lu | Speed: %.2f m/s | %.2f km/h\n", pulsePerSecond, speed_mps, speed_kmph);

    String json = "{\"speed_mps\":" + String(speed_mps, 2) + ",\"speed_kmph\":" + String(speed_kmph, 2) + "}";
    ws.textAll(json);
  }
}
