#include <WiFi.h>
#include <WebServer.h>
#include "mbedtls/aes.h"

// Wi-Fi AP Credentials
const char* ssid = "ESP32_AP";
const char* password = "12345678";

// WebServer on port 80
WebServer server(80);

// AES-128 Key (16 bytes)
const unsigned char aes_key[16] = {
    '1', '2', '3', '4', '5', '6', '7', '8',
    '9', '0', '1', '2', '3', '4', '5', '6'
};

// Buffers
unsigned char plaintext[16];
unsigned char encrypted_output[16];
unsigned char decrypted_output[16];

// Encrypt function (ECB mode)
void encryptAES(const unsigned char* input, unsigned char* output) {
    mbedtls_aes_context aes;
    mbedtls_aes_init(&aes);
    mbedtls_aes_setkey_enc(&aes, aes_key, 128);
    mbedtls_aes_crypt_ecb(&aes, MBEDTLS_AES_ENCRYPT, input, output);
    mbedtls_aes_free(&aes);
}

// Decrypt function (ECB mode)
void decryptAES(const unsigned char* input, unsigned char* output) {
    mbedtls_aes_context aes;
    mbedtls_aes_init(&aes);
    mbedtls_aes_setkey_dec(&aes, aes_key, 128);
    mbedtls_aes_crypt_ecb(&aes, MBEDTLS_AES_DECRYPT, input, output);
    mbedtls_aes_free(&aes);
}

// Web UI - Main Page
void handleRoot() {
    String html = "<!DOCTYPE html><html><head>";
    html += "<title>IndusIOT Encrypted Sender</title>";
    html += "<style>";
    html += "body { font-family: Arial, sans-serif; background: #f0f4f8; display: flex; justify-content: center; align-items: center; height: 100vh; margin: 0; }";
    html += ".container { background: white; padding: 20px; border-radius: 10px; box-shadow: 0 4px 8px rgba(0,0,0,0.1); text-align: center; }";
    html += "h1 { color: #333; }";
    html += "input[type='text'] { width: 80%; padding: 10px; margin: 10px 0; border: 1px solid #ccc; border-radius: 5px; font-size: 16px; }";
    html += "input[type='submit'] { background: #4CAF50; color: white; padding: 10px 20px; border: none; border-radius: 5px; cursor: pointer; font-size: 16px; }";
    html += "input[type='submit']:hover { background: #45a049; }";
    html += "</style></head><body>";
    html += "<div class='container'>";
    html += "<h1>IndusIOT Encrypted Sender</h1>";
    html += "<form method='POST' action='/send'>";
    html += "Message (16 chars max): <br><input type='text' name='msg' maxlength='16'><br>";
    html += "<input type='submit' value='Encrypt & Send'>";
    html += "</form></div></body></html>";
    server.send(200, "text/html", html);
}

// Handle Send Request
void handleSend() {
    String msg = server.arg("msg");
    if (msg.length() > 16) msg = msg.substring(0, 16); // Truncate
    while (msg.length() < 16) msg += " "; // Pad

    msg.getBytes(plaintext, 17); // Copy to buffer

    // Encrypt
    encryptAES(plaintext, encrypted_output);
    String encrypted_hex = "";
    for (int i = 0; i < 16; i++) {
        char hex[3];
        sprintf(hex, "%02X", encrypted_output[i]);
        encrypted_hex += hex;
    }

    // Decrypt for verification
    decryptAES(encrypted_output, decrypted_output);

    // Serial output
    Serial.println("Sender Process:");
    Serial.print("Plaintext: ");
    Serial.println((char*)plaintext);
    Serial.print("Encrypted (HEX): ");
    Serial.println(encrypted_hex);
    Serial.print("Decrypted (Verification): ");
    Serial.println((char*)decrypted_output);

    // Web response
    String response = "<!DOCTYPE html><html><head>";
    response += "<title>IndusIOT Encrypted Sender</title>";
    response += "<style>";
    response += "body { font-family: Arial, sans-serif; background: #f0f4f8; display: flex; justify-content: center; align-items: center; height: 100vh; margin: 0; }";
    response += ".container { background: white; padding: 20px; border-radius: 10px; box-shadow: 0 4px 8px rgba(0,0,0,0.1); text-align: center; }";
    response += "h1 { color: #333; }";
    response += "a { color: #4CAF50; text-decoration: none; }";
    response += "a:hover { text-decoration: underline; }";
    response += "</style></head><body>";
    response += "<div class='container'>";
    response += "<h1>Message Encrypted</h1>";
    response += "Original: " + msg + "<br>";
    response += "Encrypted: " + encrypted_hex + "<br>";
    response += "<a href='/'>Back</a>";
    response += "</div></body></html>";
    server.send(200, "text/html", response);
}

// Handle 404 (Not Found)
void handleNotFound() {
    String message = "404 - Page Not Found\n\n";
    message += "URI: ";
    message += server.uri();
    server.send(404, "text/plain", message);
}

void setup() {
    Serial.begin(115200);

    // Start ESP32 as Wi-Fi AP
    WiFi.softAP(ssid, password);
    Serial.println("WiFi AP Started! IP: " + WiFi.softAPIP().toString());

    // Web server routes
    server.on("/", handleRoot);
    server.on("/send", HTTP_POST, handleSend);
    server.onNotFound(handleNotFound); // Handle 404 errors
    server.begin();
    Serial.println("Web Server Started!");
}

void loop() {
    server.handleClient();
}