#include "esp_camera.h"
#include <WiFi.h>
#include <WebServer.h>

#define PWDN_GPIO_NUM     -1
#define RESET_GPIO_NUM    2
#define XCLK_GPIO_NUM     44
#define SIOD_GPIO_NUM     8
#define SIOC_GPIO_NUM     9

#define Y9_GPIO_NUM       6
#define Y8_GPIO_NUM       41
#define Y7_GPIO_NUM       5
#define Y6_GPIO_NUM       42
#define Y5_GPIO_NUM       4
#define Y4_GPIO_NUM       10
#define Y3_GPIO_NUM       3
#define Y2_GPIO_NUM       7
#define VSYNC_GPIO_NUM    21
#define HREF_GPIO_NUM     1
#define PCLK_GPIO_NUM     39

WebServer server(80);

framesize_t frameSize = FRAMESIZE_96X96;
pixformat_t pixelFormat = PIXFORMAT_YUV422;

void initCamera() {
  camera_config_t config;
  config.ledc_channel = LEDC_CHANNEL_0;
  config.ledc_timer = LEDC_TIMER_0;
  config.pin_d0 = Y2_GPIO_NUM;
  config.pin_d1 = Y3_GPIO_NUM;
  config.pin_d2 = Y4_GPIO_NUM;
  config.pin_d3 = Y5_GPIO_NUM;
  config.pin_d4 = Y6_GPIO_NUM;
  config.pin_d5 = Y7_GPIO_NUM;
  config.pin_d6 = Y8_GPIO_NUM;
  config.pin_d7 = Y9_GPIO_NUM;
  config.pin_xclk = XCLK_GPIO_NUM;
  config.pin_pclk = PCLK_GPIO_NUM;
  config.pin_vsync = VSYNC_GPIO_NUM;
  config.pin_href = HREF_GPIO_NUM;
  config.pin_sscb_sda = SIOD_GPIO_NUM;
  config.pin_sscb_scl = SIOC_GPIO_NUM;
  config.pin_pwdn = PWDN_GPIO_NUM;
  config.pin_reset = RESET_GPIO_NUM;
  config.xclk_freq_hz = 20000000;
  config.pixel_format = pixelFormat;
  config.frame_size = frameSize;
  config.fb_count = 1;

  esp_camera_deinit();  // Re-init camera if already initialized
  esp_err_t err = esp_camera_init(&config);
  if (err != ESP_OK) {
    Serial.printf("Camera init failed: 0x%x\n", err);
  } else {
    Serial.println("Camera Initialized");
  }
}

void handle_jpg_stream() {
  WiFiClient client = server.client();
  if (!client.connected()) return;

  client.println("HTTP/1.1 200 OK");
  client.println("Content-Type: multipart/x-mixed-replace; boundary=frame");
  client.println();

  while (client.connected()) {
    camera_fb_t * fb = esp_camera_fb_get();
    if (!fb) {
      Serial.println("Camera capture failed");
      break;
    }

    if (pixelFormat != PIXFORMAT_JPEG) {
      uint8_t * jpg_buf;
      size_t jpg_buf_len;
      if (!frame2jpg(fb, 80, &jpg_buf, &jpg_buf_len)) {
        esp_camera_fb_return(fb);
        Serial.println("JPEG compression failed");
        break;
      }

      client.printf("--frame\r\nContent-Type: image/jpeg\r\nContent-Length: %u\r\n\r\n", jpg_buf_len);
      client.write(jpg_buf, jpg_buf_len);
      client.println();
      free(jpg_buf);
    } else {
      client.printf("--frame\r\nContent-Type: image/jpeg\r\nContent-Length: %u\r\n\r\n", fb->len);
      client.write(fb->buf, fb->len);
      client.println();
    }

    esp_camera_fb_return(fb);
    delay(100);
  }
}

void handleRootPage() {
  server.send(200, "text/html", R"rawliteral(
<!DOCTYPE html>
<html>
<head>
  <title>IndusBoard Camera - Frame-by-Frame Recording</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <style>
    body {
      font-family: sans-serif;
      background: #f2f2f2;
      text-align: center;
      padding: 2rem;
    }
    h1 {
      color: #333;
    }
    .frame {
      background: #fff;
      padding: 20px;
      border-radius: 15px;
      box-shadow: 0 0 20px rgba(0,0,0,0.2);
      max-width: 90%;
      margin: auto;
    }
    .resizable {
      display: inline-block;
      position: relative;
      resize: both;
      overflow: hidden;
      border: 2px dashed #ccc;
      border-radius: 12px;
      min-width: 100px;
      min-height: 100px;
      max-width: 100%;
    }
    .resizable img {
      width: 100%;
      height: 100%;
      object-fit: contain;
    }
    button {
      margin-top: 10px;
      padding: 12px 25px;
      font-size: 1rem;
      border: none;
      border-radius: 8px;
      background: #007bff;
      color: white;
      cursor: pointer;
    }
    button:disabled {
      background: #999;
      cursor: not-allowed;
    }
  </style>
</head>
<body>
  <div class="frame">
    <h1>IndusBoard Camera</h1>
    <div class="resizable">
      <img id="stream" src="/stream">
    </div><br>
    <button onclick="startFrameByFrameRecording()">Start Recording</button>
    <button onclick="stopFrameByFrameRecording()">Stop & Download</button>
    <canvas id="canvas" style="display:none;"></canvas>
  </div>

  <script src="https://cdn.jsdelivr.net/gh/antimatter15/whammy/whammy.js"></script>
  <script>
    let frameCapture = false;
    let frames = [];
    let captureInterval;

    function startFrameByFrameRecording() {
      const img = document.getElementById('stream');
      const canvas = document.getElementById('canvas');
      const ctx = canvas.getContext('2d');

      canvas.width = img.offsetWidth;
      canvas.height = img.offsetHeight;

      frameCapture = true;
      frames = [];

      captureInterval = setInterval(() => {
        if (!frameCapture) return;
        ctx.drawImage(img, 0, 0, canvas.width, canvas.height);
        let dataURL = canvas.toDataURL('image/webp', 1.0);
        frames.push({ image: dataURL, duration: 100 });
      }, 100);

      alert("Recording started!");
    }

    function stopFrameByFrameRecording() {
      frameCapture = false;
      clearInterval(captureInterval);

      if (frames.length === 0) {
        alert("No frames captured!");
        return;
      }

      const video = new Whammy.Video(10); // 10 FPS
      frames.forEach(f => video.add(f.image, f.duration));
      const output = video.compile();
      const url = URL.createObjectURL(output);

      const a = document.createElement("a");
      a.href = url;
      a.download = "recording.webm";
      a.click();

      alert("Recording stopped and downloaded.");
    }
  </script>
</body>
</html>
  )rawliteral");
}


void handleSetMode() {
  String mode = server.arg("format");

  if (mode == "YUV422") pixelFormat = PIXFORMAT_YUV422;
  else if (mode == "RGB565") pixelFormat = PIXFORMAT_RGB565;
  else if (mode == "GRAYSCALE") pixelFormat = PIXFORMAT_GRAYSCALE;
  else pixelFormat = PIXFORMAT_JPEG;

  Serial.println("Pixel format changed to: " + mode);
  initCamera();  // reinitialize with new format
  server.sendHeader("Location", "/");
  server.send(303);
}

void startCameraServer() {
  server.on("/", HTTP_GET, handleRootPage);
  server.on("/stream", HTTP_GET, handle_jpg_stream);
  server.on("/setmode", HTTP_GET, handleSetMode);
  server.begin();
}

void setup() {
  Serial.begin(115200);
  Serial.setDebugOutput(true);
  Serial.println("\n[Starting AP Mode Camera Stream]");

  WiFi.softAP("IndusCam-AP", "12345678");
  IPAddress IP = WiFi.softAPIP();
  Serial.print("AP IP address: ");
  Serial.println(IP);

  initCamera();
  startCameraServer();
}

void loop() {
  server.handleClient();
}
