#include "esp_camera.h"
#include <WiFi.h>
#include <WebServer.h>

#define PWDN_GPIO_NUM     -1
#define RESET_GPIO_NUM    2
#define XCLK_GPIO_NUM     44
#define SIOD_GPIO_NUM     8
#define SIOC_GPIO_NUM     9

#define Y9_GPIO_NUM       6
#define Y8_GPIO_NUM       41
#define Y7_GPIO_NUM       5
#define Y6_GPIO_NUM       42
#define Y5_GPIO_NUM       4
#define Y4_GPIO_NUM       10
#define Y3_GPIO_NUM       3
#define Y2_GPIO_NUM       7
#define VSYNC_GPIO_NUM    21
#define HREF_GPIO_NUM     1
#define PCLK_GPIO_NUM     39

WebServer server(80);

framesize_t frameSize = FRAMESIZE_96X96;
pixformat_t pixelFormat = PIXFORMAT_YUV422;

void initCamera() {
  camera_config_t config;
  config.ledc_channel = LEDC_CHANNEL_0;
  config.ledc_timer = LEDC_TIMER_0;
  config.pin_d0 = Y2_GPIO_NUM;
  config.pin_d1 = Y3_GPIO_NUM;
  config.pin_d2 = Y4_GPIO_NUM;
  config.pin_d3 = Y5_GPIO_NUM;
  config.pin_d4 = Y6_GPIO_NUM;
  config.pin_d5 = Y7_GPIO_NUM;
  config.pin_d6 = Y8_GPIO_NUM;
  config.pin_d7 = Y9_GPIO_NUM;
  config.pin_xclk = XCLK_GPIO_NUM;
  config.pin_pclk = PCLK_GPIO_NUM;
  config.pin_vsync = VSYNC_GPIO_NUM;
  config.pin_href = HREF_GPIO_NUM;
  config.pin_sscb_sda = SIOD_GPIO_NUM;
  config.pin_sscb_scl = SIOC_GPIO_NUM;
  config.pin_pwdn = PWDN_GPIO_NUM;
  config.pin_reset = RESET_GPIO_NUM;
  config.xclk_freq_hz = 20000000;
  config.pixel_format = pixelFormat;
  config.frame_size = frameSize;
  config.fb_count = 1;

  esp_camera_deinit();  // Re-init camera if already initialized
  esp_err_t err = esp_camera_init(&config);
  if (err != ESP_OK) {
    Serial.printf("Camera init failed: 0x%x\n", err);
  } else {
    Serial.println("Camera Initialized");
  }
}

void handle_jpg_stream() {
  WiFiClient client = server.client();
  if (!client.connected()) return;

  client.println("HTTP/1.1 200 OK");
  client.println("Content-Type: multipart/x-mixed-replace; boundary=frame");
  client.println();

  while (client.connected()) {
    camera_fb_t * fb = esp_camera_fb_get();
    if (!fb) {
      Serial.println("Camera capture failed");
      break;
    }

    if (pixelFormat != PIXFORMAT_JPEG) {
      uint8_t * jpg_buf;
      size_t jpg_buf_len;
      if (!frame2jpg(fb, 80, &jpg_buf, &jpg_buf_len)) {
        esp_camera_fb_return(fb);
        Serial.println("JPEG compression failed");
        break;
      }

      client.printf("--frame\r\nContent-Type: image/jpeg\r\nContent-Length: %u\r\n\r\n", jpg_buf_len);
      client.write(jpg_buf, jpg_buf_len);
      client.println();
      free(jpg_buf);
    } else {
      client.printf("--frame\r\nContent-Type: image/jpeg\r\nContent-Length: %u\r\n\r\n", fb->len);
      client.write(fb->buf, fb->len);
      client.println();
    }

    esp_camera_fb_return(fb);
    delay(100);
  }
}

void handleRootPage() {
  server.send(200, "text/html", R"rawliteral(
      <!DOCTYPE html>
    <html>
    <head>
      <title>IndusBoard Camera - AP Mode</title>
      <meta name="viewport" content="width=device-width, initial-scale=1">
      <style>
        body {
          font-family: sans-serif;
          background: #f2f2f2;
          text-align: center;
          margin: 0;
          padding: 2rem;
        }
        h1 {
          color: #333;
        }
        .frame {
          background: #fff;
          padding: 20px;
          border-radius: 15px;
          box-shadow: 0 0 20px rgba(0,0,0,0.2);
          max-width: 90%;
          margin: auto;
        }
        .resizable {
          display: inline-block;
          position: relative;
          resize: both;
          overflow: hidden;
          border: 2px dashed #ccc;
          border-radius: 12px;
          min-width: 100px;
          min-height: 100px;
          max-width: 100%;
        }
        .resizable video {
          width: 100%;
          height: 100%;
          object-fit: contain;
        }
        button {
          padding: 10px 20px;
          font-size: 1rem;
          margin-top: 15px;
          border-radius: 10px;
          cursor: pointer;
        }
        select {
          padding: 10px;
          font-size: 1rem;
          margin-bottom: 20px;
          border-radius: 10px;
        }
      </style>
    </head>
    <body>
      <div class="frame">
        <h1>IndusBoard Camera</h1>
        <form action="/setmode" method="GET">
          <label>Select Pixel Format:</label>
          <select name="format" onchange="this.form.submit()">
            <option value="JPEG">JPEG</option>
            <option value="YUV422">YUV422</option>
            <option value="RGB565">RGB565</option>
            <option value="GRAYSCALE">GRAYSCALE</option>
            <option value="JPEG" selected>Default: JPEG</option>
          </select>
        </form>
        <div class="resizable">
          <video id="video" autoplay muted></video>
        </div>
        <br>
        <button onclick="toggleRecording()" id="recordBtn">Start Recording</button>
      </div>

      <script>
        let video = document.getElementById('video');
        let recordBtn = document.getElementById('recordBtn');
        let mediaRecorder;
        let recordedBlobs = [];
        let recording = false;

        // Convert MJPEG stream to MediaStream
        const streamUrl = '/stream';
        const canvas = document.createElement('canvas');
        const ctx = canvas.getContext('2d');
        const img = new Image();
        img.crossOrigin = "Anonymous";
        img.src = streamUrl;

        let fps = 10;
        img.onload = () => {
          canvas.width = img.width;
          canvas.height = img.height;
          setInterval(() => {
            ctx.drawImage(img, 0, 0, canvas.width, canvas.height);
          }, 1000 / fps);
        };

        // Capture from canvas
        const stream = canvas.captureStream(fps);
        video.srcObject = stream;

        function toggleRecording() {
          if (!recording) {
            startRecording();
          } else {
            stopRecording();
          }
        }

        function startRecording() {
          recordedBlobs = [];
          mediaRecorder = new MediaRecorder(stream, { mimeType: 'video/webm' });

          mediaRecorder.ondataavailable = (event) => {
            if (event.data && event.data.size > 0) {
              recordedBlobs.push(event.data);
            }
          };

          mediaRecorder.onstop = () => {
            const blob = new Blob(recordedBlobs, { type: 'video/webm' });
            const url = window.URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.style.display = 'none';
            a.href = url;
            a.download = 'indus_camera_recording.webm';
            document.body.appendChild(a);
            a.click();
            setTimeout(() => {
              document.body.removeChild(a);
              window.URL.revokeObjectURL(url);
            }, 100);
          };

          mediaRecorder.start();
          recordBtn.textContent = "Stop Recording";
          recording = true;
        }

        function stopRecording() {
          mediaRecorder.stop();
          recordBtn.textContent = "Start Recording";
          recording = false;
        }
      </script>
    </body>
    </html>
  )rawliteral");
}


void handleSetMode() {
  String mode = server.arg("format");

  if (mode == "YUV422") pixelFormat = PIXFORMAT_YUV422;
  else if (mode == "RGB565") pixelFormat = PIXFORMAT_RGB565;
  else if (mode == "GRAYSCALE") pixelFormat = PIXFORMAT_GRAYSCALE;
  else pixelFormat = PIXFORMAT_JPEG;

  Serial.println("Pixel format changed to: " + mode);
  initCamera();  // reinitialize with new format
  server.sendHeader("Location", "/");
  server.send(303);
}

void startCameraServer() {
  server.on("/", HTTP_GET, handleRootPage);
  server.on("/stream", HTTP_GET, handle_jpg_stream);
  server.on("/setmode", HTTP_GET, handleSetMode);
  server.begin();
}

void setup() {
  Serial.begin(115200);
  Serial.setDebugOutput(true);
  Serial.println("\n[Starting AP Mode Camera Stream]");

  WiFi.softAP("IndusCam-AP", "12345678");
  IPAddress IP = WiFi.softAPIP();
  Serial.print("AP IP address: ");
  Serial.println(IP);

  initCamera();
  startCameraServer();
}

void loop() {
  server.handleClient();
}
