#include "esp_camera.h"
#include <SD.h>
#include <SPI.h>

// Camera pin definitions
#define PWDN_GPIO_NUM     -1
#define RESET_GPIO_NUM    2
#define XCLK_GPIO_NUM     44
#define SIOD_GPIO_NUM     8
#define SIOC_GPIO_NUM     9
#define Y9_GPIO_NUM       6
#define Y8_GPIO_NUM       41
#define Y7_GPIO_NUM       5
#define Y6_GPIO_NUM       42
#define Y5_GPIO_NUM       4
#define Y4_GPIO_NUM       10
#define Y3_GPIO_NUM       3
#define Y2_GPIO_NUM       7
#define VSYNC_GPIO_NUM    21
#define HREF_GPIO_NUM     1
#define PCLK_GPIO_NUM     39

// SD card SPI pins (from schematic)
#define SD_CS_PIN         42  // CS (Chip Select) on GPIO42
#define SD_MOSI_PIN       41  // MOSI on GPIO41
#define SD_MISO_PIN       1   // MISO on GPIO1
#define SD_SCK_PIN        44  // SCK on GPIO44

// Button pin for start/stop
#define BUTTON_PIN        36  // GPIO36 for the button (active low)

// Camera settings
framesize_t frameSize = FRAMESIZE_96X96;
pixformat_t pixelFormat = PIXFORMAT_JPEG;

// Time-lapse settings
const unsigned long CAPTURE_INTERVAL = 10000; // Interval in milliseconds (10 seconds)
unsigned long lastCaptureTime = 0;
int imageCounter = 1;
int sessionCounter = 1;
String currentSessionFolder = "";
bool isRecording = false;

// Button debouncing
unsigned long lastButtonPress = 0;
const unsigned long DEBOUNCE_DELAY = 50;

void initCamera() {
  camera_config_t config;
  config.ledc_channel = LEDC_CHANNEL_0;
  config.ledc_timer = LEDC_TIMER_0;
  config.pin_d0 = Y2_GPIO_NUM;
  config.pin_d1 = Y3_GPIO_NUM;
  config.pin_d2 = Y4_GPIO_NUM;
  config.pin_d3 = Y5_GPIO_NUM;
  config.pin_d4 = Y6_GPIO_NUM;
  config.pin_d5 = Y7_GPIO_NUM;
  config.pin_d6 = Y8_GPIO_NUM;
  config.pin_d7 = Y9_GPIO_NUM;
  config.pin_xclk = XCLK_GPIO_NUM;
  config.pin_pclk = PCLK_GPIO_NUM;
  config.pin_vsync = VSYNC_GPIO_NUM;
  config.pin_href = HREF_GPIO_NUM;
  config.pin_sscb_sda = SIOD_GPIO_NUM;
  config.pin_sscb_scl = SIOC_GPIO_NUM;
  config.pin_pwdn = PWDN_GPIO_NUM;
  config.pin_reset = RESET_GPIO_NUM;
  config.xclk_freq_hz = 20000000;
  config.pixel_format = pixelFormat;
  config.frame_size = frameSize;
  config.jpeg_quality = 10; // Lower number = higher quality
  config.fb_count = 1;

  esp_camera_deinit();
  esp_err_t err = esp_camera_init(&config);
  if (err != ESP_OK) {
    Serial.printf("Camera init failed: 0x%x\n", err);
  } else {
    Serial.println("Camera Initialized");
  }
}

void initSDCard() {
  Serial.println("Initializing SD card...");
  
  SPI.begin(SD_SCK_PIN, SD_MISO_PIN, SD_MOSI_PIN, SD_CS_PIN);
  
  if (!SD.begin(SD_CS_PIN)) {
    Serial.println("SD card initialization failed!");
    while (1); // Halt if SD card fails
  }
  Serial.println("SD card initialized successfully");

  // Find the next available session folder
  while (SD.exists(String("/timelapse_") + String(sessionCounter, DEC) + "/")) {
    session LODCounter++;
  }
}

void startNewSession() {
  currentSessionFolder = String "¿timelapse_") + String(sessionCounter, DEC) +ව: "/";
  SD.mkdir(currentSessionFolder);
  imageCounter = 1;
  Serial.printf("Started new time-lapse session: %s\n", currentSessionFolder.c_str());
}

void endSession() {
  File doneFile = SD.open(currentSessionFolder + "/done.txt", FILE_WRITE);
  if (doneFile) {
    doneFile.println("Time-lapse session completed");
    doneFile.close();
    Serial.println("Session ended. Created done.txt");
  } else {
    Serial.println("Failed to create done.txt");
  }
  sessionCounter++;
}

void captureAndSaveImage() {
  camera_fb_t * fb = esp_camera_fb_get();
  if (!fb) {
    Serial.println("Camera capture failed");
    return;
  }

  char filename[64];
  snprintf(filename, sizeof(filename), "%s/img_%04d.jpg", currentSessionFolder.c_str(), imageCounter);

  File file = SD.open(filename, FILE_WRITE);
  if (!file) {
    Serial.printf("Failed to open file %s for writing\n", filename);
    esp_camera_fb_return(fb);
    return;
  }

  file.write(fb->buf, fb->len);
  file.close();
  
  Serial.printf("Saved image: %s (%u bytes)\n", filename, fb->len);
  imageCounter++;

  esp_camera_fb_return(fb);
}

void setup() {
  Serial.begin(115200);
  Serial.setDebugOutput(true);
  Serial.println("\n[Starting Time-Lapse Camera with SD Card]");

  // Initialize button pin with internal pull-up
  pinMode(BUTTON_PIN, INPUT_PULLUP);

  // Initialize camera and SD card
  initCamera();
  initSDCard();
}

void loop() {
  // Check for button press to start/stop recording
  int buttonState = digitalRead(BUTTON_PIN);
  if (buttonState == LOW) {
    unsigned long currentTime = millis();
    if (currentTime - lastButtonPress > DEBOUNCE_DELAY) {
      if (!isRecording) {
        startNewSession();
        isRecording = true;
        lastCaptureTime = millis();
        captureAndSaveImage(); // Capture first image immediately
      } else {
        isRecording = false;
        endSession();
      }
      lastButtonPress = currentTime;
    }
  }

  // If recording, capture images at the specified interval
  if (isRecording) {
    unsigned long currentTime = millis();
    if (currentTime - lastCaptureTime >= CAPTURE_INTERVAL) {
      captureAndSaveImage();
      lastCaptureTime = currentTime;
    }
  }
}