#include <SPI.h>
#include <MFRC522.h>
#include <WiFi.h>
#include <WebServer.h>
#include <NTPClient.h>
#include <WiFiUdp.h>

// Define pins for MFRC522
#define RST_PIN 6
#define SS_PIN 34
#define MOSI_PIN 35
#define MISO_PIN 37
#define SCK_PIN 36

// Create MFRC522 instance
MFRC522 mfrc522(SS_PIN, RST_PIN);

// Create web server instance
WebServer server(80);

// Define Wi-Fi credentials
const char* ssid = "Your_SSID";
const char* password = "Your_PASSWORD";

// NTP setup
WiFiUDP ntpUDP;
NTPClient timeClient(ntpUDP, "pool.ntp.org", 19800, 60000); // IST offset: +5:30 (19800 seconds)

// Define dynamic arrays for users and their RFID UIDs
String* users = new String[10] {"James", "Anna", "John", "Emily", "Michael", "Sarah", "David", "Laura", "Daniel", "Sophia"};
String* uids = new String[10] {"UID1", "UID2", "UID3", "UID4", "UID5", "UID6", "UID7", "UID8", "UID9", "UID10"};
String* attendance = new String[10]; // Store detailed attendance records
int maxUsers = 10;
int userCount = 10; // Initial count of users

// Structure to hold attendance data
struct AttendanceRecord {
  String uid;
  String name;
  String time;
  String date;
  String day;
};

// Array to store attendance records
AttendanceRecord attendanceRecords[100];
int attendanceCount = 0;

void setup() {
  Serial.begin(115200);
  SPI.begin(SCK_PIN, MISO_PIN, MOSI_PIN, SS_PIN); // Initiate SPI bus with custom pins
  mfrc522.PCD_Init();   // Initiate MFRC522
  WiFi.begin(ssid, password);
  // Wait for the Wi-Fi connection
  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    Serial.print(".");
  }
  Serial.println("Connected to WiFi");
  timeClient.begin();
  timeClient.update();
  // Define routes for the web server
  server.on("/", HTTP_GET, handleRoot);
  server.on("/attendance", HTTP_GET, handleAttendance);
  server.on("/entry", HTTP_GET, handleEntry);
  server.on("/save", HTTP_GET, handleSave);
  server.on("/download", HTTP_GET, handleDownload);
  server.begin();
}

void loop() {
  server.handleClient();

  if (!mfrc522.PICC_IsNewCardPresent()) {
    return;
  }

  if (!mfrc522.PICC_ReadCardSerial()) {
    return;
  }

  String uid = "";
  for (byte i = 0; i < mfrc522.uid.size; i++) {
    uid += String(mfrc522.uid.uidByte[i] < 0x10 ? "0" : "") + String(mfrc522.uid.uidByte[i], HEX);
  }
  uid.toUpperCase();
  Serial.println(uid);
  bool recognized = false;
  for (int i = 0; i < userCount; i++) {
    if (uid == uids[i]) {
      timeClient.update();
      String time = timeClient.getFormattedTime();
      String date = getFormattedDate(timeClient.getEpochTime());
      String day = getDayOfWeek(timeClient.getDay());
      if (attendanceRecords[i].uid.isEmpty() || attendanceRecords[i].time.indexOf("Punch-Out") == -1) {
        attendanceRecords[i] = {uid, users[i], "Punch-In: " + time, date, day};
        Serial.println(users[i] + " punched in at " + time);
      } else {
        attendanceRecords[i].time += "\nPunch-Out: " + time;
        Serial.println(users[i] + " punched out at " + time);
      }
      recognized = true;
      break;
    }
  }

  if (recognized) {
    server.send(200, "text/plain", "Attendance updated.");
  } else {
    server.sendHeader("Location", "/entry?uid=" + uid);
    server.send(302, "text/plain", "Redirecting to entry page...");
  }

  mfrc522.PICC_HaltA();
}

void handleRoot() {
  server.send(200, "text/html", "<html><head><title>RFID Attendance System</title></head><body><h1>RFID Attendance System</h1><a href='/attendance'>View Attendance</a></body></html>");
}

void handleAttendance() {
  timeClient.update();
  String html = "<html><head><title>Attendance Records</title><style>body { font-family: Arial, sans-serif; background-color: #f4f4f4; margin: 0; padding: 0; } h1 { text-align: center; color: #333; } table { width: 80%; margin: 20px auto; border-collapse: collapse; } th, td { padding: 10px; border: 1px solid #ddd; } th { background-color: #f2f2f2; } tr:nth-child(even) { background-color: #f9f9f9; } tr:hover { background-color: #ddd; } button { background-color: #4CAF50; color: white; padding: 10px 20px; border: none; cursor: pointer; } button:hover { background-color: #45a049; }</style></head><body><h1>Attendance Records</h1><table><tr><th>UID</th><th>Name</th><th>Time</th><th>Date</th><th>Day</th></tr>";
  for (int i = 0; i < userCount; i++) {
    if (!attendanceRecords[i].uid.isEmpty()) {
      html += "<tr><td>" + attendanceRecords[i].uid + "</td><td>" + attendanceRecords[i].name + "</td><td>" + attendanceRecords[i].time + "</td><td>" + attendanceRecords[i].date + "</td><td>" + attendanceRecords[i].day + "</td></tr>";
    }
  }
  html += "</table><button onclick=\"window.location.href='/download'\">Download Today's Attendance</button></body></html>";
  server.send(200, "text/html", html);
}

void handleEntry() {
  String uid = server.arg("uid");
  String html = "<html><head><title>New Entry</title><style>body { font-family: Arial, sans-serif; margin: 20px; } input { padding: 5px; } button { background-color: #4CAF50; color: white; padding: 10px; border: none; cursor: pointer; } button:hover { background-color: #45a049; }</style></head><body><h2>Unknown UID: " + uid + "</h2><form action='/save'><label>Name: <input type='text' name='name' required></label><input type='hidden' name='uid' value='" + uid + "'><button type='submit'>Save</button></form></body></html>";
  server.send(200, "text/html", html);
}

void handleSave() {
  if (server.hasArg("uid") && server.hasArg("name")) {
    String uid = server.arg("uid");
    String name = server.arg("name");
    if (userCount < maxUsers) {
      String* newUsers = new String[maxUsers];
      String* newUids = new String[maxUsers];
      String* newAttendance = new String[maxUsers];
      AttendanceRecord* newRecords = new AttendanceRecord[maxUsers];
      for (int i = 0; i < userCount; i++) {
        newUsers[i] = users[i];
        newUids[i] = uids[i];
        newAttendance[i] = attendance[i];
        newRecords[i] = attendanceRecords[i];
      }
      newUsers[userCount] = name;
      newUids[userCount] = uid;
      newAttendance[userCount] = "";
      newRecords[userCount] = {uid, name, "", "", ""};
      delete[] users;
      delete[] uids;
      delete[] attendance;
      // Copy newRecords to attendanceRecords individually since it's an array
      for (int i = 0; i < maxUsers; i++) {
        attendanceRecords[i] = newRecords[i];
      }
      delete[] newRecords; // Delete the dynamically allocated array after copying
      users = newUsers;
      uids = newUids;
      attendance = newAttendance;
      userCount++;
      Serial.println("New user added: " + name + " (UID: " + uid + ")");
    }
    server.sendHeader("Location", "/attendance");
    server.send(302, "text/plain", "Redirecting to attendance page...");
  }
}

void handleDownload() {
  timeClient.update();
  String csv = "UID,Name,Time,Date,Day\n";
  for (int i = 0; i < userCount; i++) {
    if (!attendanceRecords[i].uid.isEmpty()) {
      String timeStr = attendanceRecords[i].time;
      timeStr.replace("\n", " "); // Replace newlines with spaces
      csv += attendanceRecords[i].uid + "," + attendanceRecords[i].name + "," + timeStr + "," + attendanceRecords[i].date + "," + attendanceRecords[i].day + "\n";
    }
  }
  server.sendHeader("Content-Type", "text/csv");
  server.sendHeader("Content-Disposition", "attachment; filename=attendance_" + getFormattedDate(timeClient.getEpochTime()) + ".csv");
  server.send(200, "text/csv", csv);
}

String getFormattedDate(unsigned long epoch) {
  time_t rawtime = epoch;
  struct tm* timeinfo = localtime(&rawtime);
  char buffer[11];
  strftime(buffer, sizeof(buffer), "%Y-%m-%d", timeinfo);
  return String(buffer);
}

String getDayOfWeek(uint8_t day) {
  const char* days[] = {"Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday"};
  return days[day];
}