#include <TFT_eSPI.h>
#include <CST816S.h>
#include "Free_Fonts.h"
#include <Wire.h>
#include <TimeLib.h>  // For real-time clock functionality
#include <RDA5807M.h> // RDA5807M FM Radio library

// Initialize TFT and Touch
TFT_eSPI tft = TFT_eSPI();
CST816S touch(5, 6, 10, 7);

// Initialize RDA5807M
RDA5807M radio;

// Radio state
bool isPlaying = false;
float currentFrequency = 87.5; // Starting frequency (MHz)
bool onInitialScreen = true;

// Button positions and radius
const int buttonRadius = 30;
const int playPauseButtonX = 80;
const int playPauseButtonY = 180;
const int nextButtonX = 160;
const int nextButtonY = 180;

void setup() {
  Serial.begin(115200);
  Wire.begin(); // Initialize I2C for RDA5807M

  // Initialize TFT
  pinMode(TFT_BL, OUTPUT);
  digitalWrite(TFT_BL, HIGH);
  tft.begin();
  tft.fillScreen(TFT_BLACK);
  tft.setTextColor(TFT_WHITE, TFT_BLACK);
  tft.setFreeFont(FSB9);

  // Initialize touch
  touch.begin();

  // Initialize radio
  radio.init();
  radio.setBandFrequency(RADIO_BAND_FM, currentFrequency * 100); // Set initial frequency (in kHz)
  radio.setVolume(8); // Set volume (0-15)
  radio.setMute(true); // Mute by default (not playing)

  // Display initial screen
  displayInitialScreen();
}

void loop() {
  if (onInitialScreen) {
    delay(4000); // Show initial screen for 4 seconds
    onInitialScreen = false;
    tft.fillScreen(TFT_BLACK);
    drawRadioInterface();
  }

  if (touch.available()) {
    int x = touch.data.x;
    int y = touch.data.y;

    if (isPlayPauseButtonTouched(x, y)) {
      togglePlayPause();
    }
    if (isNextButtonTouched(x, y)) {
      nextChannel();
    }
  }
}

void displayInitialScreen() {
  // Draw time in the center
  setTime(10, 30, 0, 1, 1, 2024); // Demo time
  char timeString[9];
  sprintf(timeString, "%02d:%02d:%02d", hour(), minute(), second());
  int centerX = 120;
  int centerY = 120;
  tft.setCursor(centerX - 40, centerY - 10);
  tft.setTextColor(TFT_WHITE, TFT_BLACK);
  tft.setFreeFont(FSB9);
  tft.print(timeString);

  // Draw battery percentage
  int batteryX = 10;
  int batteryY = 180;
  int batteryWidth = 50;
  int batteryHeight = 10;
  int batteryCharge = 75; // Example
  tft.drawRect(batteryX, batteryY, batteryWidth, batteryHeight, TFT_WHITE);
  tft.fillRect(batteryX + 1, batteryY + 1, (batteryWidth - 2) * batteryCharge / 100, batteryHeight - 2, TFT_GREEN);

  // Draw signal strength
  int signalX = 180;
  int signalY = 180;
  int signalWidth = 8;
  int signalHeight = 10;
  int signalBars = 3; // Fixed signal strength (3 bars)
  for (int i = 0; i < signalBars && i < 5; i++) {
    tft.fillRect(signalX, signalY + (signalHeight - (i + 1) * 2), signalWidth, (i + 1) * 2, TFT_WHITE);
    signalX -= signalWidth + 2;
  }

  // Draw initial frequency
  tft.setCursor(70, 50);
  tft.setTextColor(TFT_GREEN, TFT_BLACK);
  char freqString[10];
  sprintf(freqString, "%.1f MHz", currentFrequency);
  tft.print(freqString);
}

void drawRadioInterface() {
  // Display current frequency
  tft.setCursor(70, 50);
  tft.setTextColor(TFT_GREEN, TFT_BLACK);
  char freqString[10];
  sprintf(freqString, "%.1f MHz", currentFrequency);
  tft.print(freqString);

  // Display play/pause status
  tft.setCursor(80, 100);
  tft.setTextColor(TFT_WHITE, TFT_BLACK);
  tft.print(isPlaying ? "Playing" : "Paused");

  // Draw buttons
  drawButtons();
}

void drawButtons() {
  // Play/Pause Button
  tft.fillCircle(playPauseButtonX, playPauseButtonY, buttonRadius, isPlaying ? TFT_YELLOW : TFT_GREEN);
  tft.setTextColor(TFT_WHITE, isPlaying ? TFT_YELLOW : TFT_GREEN);
  tft.setCursor(playPauseButtonX - 20, playPauseButtonY - 7);
  tft.print(isPlaying ? "Pause" : "Play");

  // Next Channel Button
  tft.fillCircle(nextButtonX, nextButtonY, buttonRadius, TFT_BLUE);
  tft.setTextColor(TFT_WHITE, TFT_BLUE);
  tft.setCursor(nextButtonX - 15, nextButtonY - 7);
  tft.print("Next");
}

bool isPlayPauseButtonTouched(int x, int y) {
  int dx = x - playPauseButtonX;
  int dy = y - playPauseButtonY;
  return (dx * dx + dy * dy <= buttonRadius * buttonRadius);
}

bool isNextButtonTouched(int x, int y) {
  int dx = x - nextButtonX;
  int dy = y - nextButtonY;
  return (dx * dx + dy * dy <= buttonRadius * buttonRadius);
}

void togglePlayPause() {
  isPlaying = !isPlaying;
  radio.setMute(!isPlaying);
  tft.fillScreen(TFT_BLACK);
  drawRadioInterface();
}

void nextChannel() {
  currentFrequency += 0.1; // Increment by 0.1 MHz
  if (currentFrequency > 108.0) {
    currentFrequency = 87.5; // Wrap around to start
  }
  radio.setBandFrequency(RADIO_BAND_FM, currentFrequency * 100); // Set frequency in kHz
  tft.fillScreen(TFT_BLACK);
  drawRadioInterface();
}