import tkinter as tk
from tkinter import filedialog, messagebox
import subprocess
import os
import signal

class FMTransmitterGUI:
    def __init__(self, root):
        self.root = root
        self.root.title("Raspberry Pi FM Transmitter")
        self.process = None

        # Main frame to organize elements
        main_frame = tk.Frame(root, padx=10, pady=10, bg="white")
        main_frame.pack(expand=True)

        # Title
        tk.Label(main_frame, text="FM Transmitter Control", font=("Arial", 14, "bold")).pack(pady=5)

        # File selection
        file_frame = tk.Frame(main_frame, bg="white")
        file_frame.pack(pady=5, fill=tk.X)
        tk.Label(file_frame, text="Select WAV File:", bg="white").pack(side=tk.LEFT)
        self.file_path = tk.StringVar()
        tk.Entry(file_frame, textvariable=self.file_path, width=40, fg="black", bg="white").pack(side=tk.LEFT, padx=5)
        tk.Button(file_frame, text="Browse", command=self.browse_file).pack(side=tk.LEFT)

        # Frequency input
        freq_frame = tk.Frame(main_frame, bg="white")
        freq_frame.pack(pady=5, fill=tk.X)
        tk.Label(freq_frame, text="Frequency (MHz, 87.5-108.0):", bg="white").pack(side=tk.LEFT)
        self.frequency = tk.DoubleVar(value=100.0)
        freq_entry = tk.Entry(freq_frame, textvariable=self.frequency, width=10, fg="black", bg="white")
        freq_entry.pack(side=tk.LEFT, padx=5)
        freq_entry.bind("<FocusOut>", self.validate_frequency)

        # Bandwidth input
        bw_frame = tk.Frame(main_frame, bg="white")
        bw_frame.pack(pady=5, fill=tk.X)
        tk.Label(bw_frame, text="Bandwidth (kHz, e.g., 100):", bg="white").pack(side=tk.LEFT)
        self.bandwidth = tk.DoubleVar(value=100.0)
        tk.Entry(bw_frame, textvariable=self.bandwidth, width=10, fg="black", bg="white").pack(side=tk.LEFT, padx=5)

        # Loop playback checkbox
        self.loop = tk.BooleanVar()
        tk.Checkbutton(main_frame, text="Loop Playback", bg="white").pack(pady=5)

        # Start/Stop buttons
        button_frame = tk.Frame(main_frame, bg="white")
        button_frame.pack(pady=5)
        tk.Button(button_frame, text="Start Transmission", command=self.start_transmission).pack(side=tk.LEFT, padx=5)
        tk.Button(button_frame, text="Stop Transmission", command=self.stop_transmission).pack(side=tk.LEFT, padx=5)

        # Status label
        self.status = tk.StringVar(value="Ready")
        tk.Label(main_frame, textvariable=self.status, bg="white").pack(pady=10)

        # Ensure process cleanup on window close
        self.root.protocol("WM_DELETE_WINDOW", self.on_closing)

    def validate_frequency(self, event=None):
        try:
            freq = self.frequency.get()
            if not (87.5 <= freq <= 108.0):
                messagebox.showwarning("Invalid Frequency", "Frequency must be between 87.5 and 108.0 MHz")
                self.frequency.set(100.0)
        except tk.TclError:
            messagebox.showwarning("Invalid Input", "Please enter a valid number")
            self.frequency.set(100.0)

    def browse_file(self):
        file = filedialog.askopenfilename(filetypes=[("WAV files", "*.wav")])
        if file:
            self.file_path.set(file)

    def start_transmission(self):
        try:
            freq = self.frequency.get()
            if not (87.5 <= freq <= 108.0):
                messagebox.showerror("Error", "Frequency must be between 87.5 and 108.0 MHz")
                return
        except tk.TclError:
            messagebox.showerror("Error", "Invalid frequency input")
            return
        if not self.file_path.get():
            messagebox.showerror("Error", "Please select a WAV file")
            return
        if not os.path.exists(self.file_path.get()):
            messagebox.showerror("Error", "Selected file does not exist")
            return
        bw = self.bandwidth.get()
        if bw <= 0:
            messagebox.showerror("Error", "Bandwidth must be positive")
            return

        self.stop_transmission()

        cmd = ["sudo", "./fm_transmitter", "-f", str(freq), "-b", str(bw)]
        if self.loop.get():
            cmd.append("-r")
        cmd.append(self.file_path.get())

        try:
            self.process = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE, text=True)
            self.status.set(f"Transmitting {os.path.basename(self.file_path.get())} at {freq} MHz")
        except Exception as e:
            messagebox.showerror("Error", f"Failed to start transmission: {str(e)}")
            self.status.set("Error starting transmission")

    def stop_transmission(self):
        if self.process:
            os.kill(self.process.pid, signal.SIGTERM)
            self.process = None
            self.status.set("Transmission stopped")
        else:
            self.status.set("No transmission running")

    def on_closing(self):
        self.stop_transmission()
        self.root.destroy()

if __name__ == "__main__":
    root = tk.Tk()
    app = FMTransmitterGUI(root)
    root.geometry("400x350")
    root.mainloop()