#include <Wire.h>
#include <LSM303AGR_ACC_Sensor.h>
#include <WiFi.h>
#include <WebServer.h>


// AP credentials
const char* apSSID = "SoilMoistureAP";
const char* apPassword = "12345678"; // Minimum 8 characters


// Sensor pins and definitions
#define LIGHT_SENSOR_PIN 2 // XYC-PT19C-L1 connected to pin 2
#define SOIL_SENSOR_PIN 4  // Soil moisture sensor connected to pin 4
#define DEV_I2C Wire       // Define I2C bus
LSM303AGR_ACC_Sensor Acc(&DEV_I2C);


// Web server on port 80
WebServer server(80);


// Data storage for 30 land parts
float soilMoisture[30];
float lightLux[30];
float temperature[30];
int currentPart = 0;
bool readingsTaken[30] = {false};


// Land area input (default 1000 sqm, updated via webpage)
float landArea = 1000.0;


void setup() {
 Serial.begin(115200);
 pinMode(LIGHT_SENSOR_PIN, INPUT);
 pinMode(SOIL_SENSOR_PIN, INPUT);
  // Initialize I2C and accelerometer
 DEV_I2C.begin();
 Acc.begin();
 Acc.EnableTemperatureSensor();


 // Set up Access Point
 WiFi.softAP(apSSID, apPassword);
 IPAddress IP = WiFi.softAPIP();
 Serial.print("AP SSID: ");
 Serial.println(apSSID);
 Serial.print("AP IP address: ");
 Serial.println(IP);


 // Web server routes
 server.on("/", handleRoot);
 server.on("/capture", handleCapture);
 server.on("/setArea", handleSetArea);
 server.on("/liveData", handleLiveData);
 server.begin();
}


void loop() {
 server.handleClient();
 delay(10);
}


void handleRoot() {
 String html = "<!DOCTYPE html><html><head>";
 html += "<title>Soil Moisture Mapping</title>";
 html += "<style>";
 html += "body { font-family: Arial, sans-serif; background-color: #ffffff; margin: 0; padding: 20px; }";
 html += ".container { max-width: 1200px; margin: auto; }";
 html += ".card { background: #f4f4f4; border-radius: 8px; padding: 20px; margin: 10px; display: inline-block; width: 200px; }";
 html += ".land-grid { display: grid; grid-template-columns: repeat(5, 1fr); gap: 2px; }";
 html += ".land-part { width: 100px; height: 100px; border: 1px solid lightgray; }";
 html += ".btn { background: #4CAF50; color: white; padding: 10px 20px; border: none; border-radius: 4px; cursor: pointer; }";
 html += ".btn:hover { background: #45a049; }";
 html += ".input-field { padding: 8px; margin: 10px 0; }";
 html += "</style></head><body>";
 html += "<div class='container'>";
 html += "<h1>Soil Moisture Mapping</h1>";
 html += "<div class='card'>";
 html += "<h3>Live Sensor Readings</h3>";
 html += "<p>Soil Moisture: <span id='liveMoisture'>0</span>%</p>";
 html += "<p>Light: <span id='liveLight'>0</span> Lux</p>";
 html += "<p>Temperature: <span id='liveTemp'>0</span> °C</p>";
 html += "</div>";
 html += "<div class='card'>";
 html += "<h3>Captured Data</h3>";
 html += "<p>Average Moisture: <span id='avgMoisture'>0</span>%</p>";
 html += "<p>Driest Part: <span id='driest'>None</span></p>";
 html += "</div>";
 html += "<div>";
 html += "<input type='number' id='landArea' class='input-field' placeholder='Enter land area (sqm)' value='" + String(landArea) + "'>";
 html += "<button class='btn' onclick='setArea()'>Set Area</button>";
 html += "<button class='btn' id='captureBtn' onclick='captureData()'>Capture Data for Part " + String(currentPart + 1) + "</button>";
 html += "</div>";
 html += "<h2>Land Visualization</h2>";
 html += "<div class='land-grid' id='landGrid'>";
 for (int i = 0; i < 30; i++) {
   html += "<div class='land-part' id='part" + String(i) + "' style='background-color: lightblue;'></div>";
 }
 html += "</div>";
 html += "<script>";
 html += "function captureData() { fetch('/capture').then(response => response.json()).then(data => {";
 html += "document.getElementById('avgMoisture').innerText = data.avgMoisture.toFixed(2);";
 html += "document.getElementById('driest').innerText = data.driestPart;";
 html += "document.getElementById('captureBtn').innerText = 'Capture Data for Part ' + data.nextPart;";
 html += "for (let i = 0; i < 30; i++) {";
 html += "let moisture = data.moistures[i] || 0;";
 html += "let blue = Math.min(255, moisture * 2.55);";
 html += "document.getElementById('part' + i).style.backgroundColor = `rgb(0, ${255-blue}, 255)`;";
 html += "}})}";
 html += "function setArea() {";
 html += "let area = document.getElementById('landArea').value;";
 html += "fetch('/setArea?area=' + area);}";
 html += "function updateLiveData() {";
 html += "fetch('/liveData').then(response => response.json()).then(data => {";
 html += "document.getElementById('liveMoisture').innerText = data.soilMoisture.toFixed(2);";
 html += "document.getElementById('liveLight').innerText = data.light.toFixed(2);";
 html += "document.getElementById('liveTemp').innerText = data.temp.toFixed(2);";
 html += "})}";
 html += "setInterval(updateLiveData, 1000);";
 html += "</script>";
 html += "</div></body></html>";
 server.send(200, "text/html", html);
}


void handleLiveData() {
 // Read sensors
 int soilAdc = analogRead(SOIL_SENSOR_PIN);
 float currentMoisture = map(soilAdc, 0, 4095, 100, 0); // 0 (wet) to 100 (dry)
 int lightAdc = analogRead(LIGHT_SENSOR_PIN);
 float currentLight = (lightAdc / 4095.0) * 1000.0;
 float currentTemp;
 Acc.GetTemperature(&currentTemp);


 // Send JSON response
 String json = "{";
 json += "\"soilMoisture\":" + String(currentMoisture, 2) + ",";
 json += "\"light\":" + String(currentLight, 2) + ",";
 json += "\"temp\":" + String(currentTemp, 2);
 json += "}";
 server.send(200, "application/json", json);
}


void handleCapture() {
 if (currentPart < 30) {
   // Read sensors
   int soilAdc = analogRead(SOIL_SENSOR_PIN);
   soilMoisture[currentPart] = map(soilAdc, 0, 4095, 100, 0); // 0 (wet) to 100 (dry)
   int lightAdc = analogRead(LIGHT_SENSOR_PIN);
   lightLux[currentPart] = (lightAdc / 4095.0) * 1000.0;
   Acc.GetTemperature(&temperature[currentPart]);
   readingsTaken[currentPart] = true;
   currentPart++;
 }


 // Calculate average moisture and driest part
 float totalMoisture = 0;
 float maxDryness = 0;
 int driestPart = 0;
 int count = 0;
 for (int i = 0; i < 30; i++) {
   if (readingsTaken[i]) {
     totalMoisture += soilMoisture[i];
     count++;
     if (soilMoisture[i] > maxDryness) {
       maxDryness = soilMoisture[i];
       driestPart = i + 1;
     }
   }
 }
 float avgMoisture = count > 0 ? totalMoisture / count : 0;


 // Send JSON response
 String json = "{";
 json += "\"avgMoisture\":" + String(avgMoisture, 2) + ",";
 json += "\"driestPart\":" + String(driestPart) + ",";
 json += "\"nextPart\":" + String(currentPart < 30 ? currentPart + 1 : 1) + ",";
 json += "\"moistures\":[";
 for (int i = 0; i < 30; i++) {
   json += String(soilMoisture[i], 2);
   if (i < 29) json += ",";
 }
 json += "]}";
 server.send(200, "application/json", json);
}


void handleSetArea() {
 if (server.hasArg("area")) {
   landArea = server.arg("area").toFloat();
 }
 server.send(200, "text/plain", "Area updated");
}

