
// Description:
// ------------
// Simplified firmware for Pocket Radio using a single switch on PA3.
// Quick press: Seek next channel.
// Press and hold for 2s: Seek previous channel.
// Long press for 10s: Increase volume.
// Long press for 30s: Decrease volume.
//
// Wiring:
// -------
//                           +-\/-+
//                     Vdd  1|°   |8  GND
//              Switch --- PA6  2|    |7  PA3 -------- Switch to GND
//                     RXD PA7  3|    |6  PA0 AIN0 UPDI --- UPDI
//                     SDA PA1  4|    |5  PA2 AIN2 SCL ---- (Unused)
//                           +----+
//
// Compilation Settings:
// ---------------------
// Core:    megaTinyCore (https://github.com/SpenceKonde/megaTinyCore)
// Board:   ATtiny412/402/212/202
// Chip:    ATtiny402 or ATtiny412
// Clock:   1 MHz internal
//http://drazzy.com/package_drazzy.com_index.json

// Leave the rest on default settings. Use a programmer that works with 3.3V.
// Don't forget to "Burn bootloader". Compile and upload the code.

// ===================================================================================
// Libraries, Definitions and Macros
// ===================================================================================

// Libraries
#include <avr/io.h>                 // for GPIO
#include <avr/eeprom.h>             // for storing user settings into EEPROM
#include <util/delay.h>             // for delays

// Pin assignments
#define PIN_SWITCH      PA3           // Single switch pin

// Firmware parameters
#define EEPROM_IDENT    0x6CE7        // to identify if EEPROM was written by this program

// Variables
uint16_t  channel;                  // 0 .. 1023   current channel
uint8_t   volume = 5;               // 0 .. 15     current volume

// Pin manipulation macros
enum {PA0, PA1, PA2, PA3, PA4, PA5, PA6, PA7};      // enumerate pin designators
#define pinInput(x)       VPORTA.DIR &= ~(1<<(x))   // set pin to INPUT
#define pinOutput(x)      VPORTA.DIR |=  (1<<(x))   // set pin to OUTPUT
#define pinLow(x)         VPORTA.OUT &= ~(1<<(x))   // set pin to LOW
#define pinHigh(x)        VPORTA.OUT |=  (1<<(x))   // set pin to HIGH
#define pinToggle(x)      VPORTA.IN  |=  (1<<(x))   // TOGGLE pin
#define pinRead(x)        (VPORTA.IN &   (1<<(x)))  // READ pin
#define pinPullup(x)      (&PORTA.PIN0CTRL)[x] |= PORT_PULLUPEN_bm  // enable pullup

// ===================================================================================
// I2C Master Implementation (Software Bit-Banging)
// ===================================================================================

// I2C macros
#define I2C_SDA_HIGH()  pinInput(PA1)       // release SDA   -> pulled HIGH by resistor
#define I2C_SDA_LOW()   pinOutput(PA1)      // SDA as output -> pulled LOW  by MCU
#define I2C_SCL_HIGH()  pinInput(PA2)       // release SCL   -> pulled HIGH by resistor
#define I2C_SCL_LOW()   pinOutput(PA2)      // SCL as output -> pulled LOW  by MCU
#define I2C_SDA_READ()  pinRead(PA1)        // read SDA line
#define I2C_CLOCKOUT()  I2C_SCL_HIGH();I2C_SCL_LOW()  // clock out

// I2C transmit one data byte to the slave, ignore ACK bit, no clock stretching allowed
void I2C_write(uint8_t data) {
  for(uint8_t i=8; i; i--, data<<=1) {
    (data&0x80)?I2C_SDA_HIGH():I2C_SDA_LOW();
    I2C_CLOCKOUT();
  }
  I2C_SDA_HIGH();
  I2C_CLOCKOUT();
}

// I2C start transmission
void I2C_start(uint8_t addr) {
  I2C_SDA_LOW();
  I2C_SCL_LOW();
  I2C_write(addr);
}

// I2C restart transmission
void I2C_restart(uint8_t addr) {
  I2C_SDA_HIGH();
  I2C_SCL_HIGH();
  I2C_start(addr);
}

// I2C stop transmission
void I2C_stop(void) {
  I2C_SDA_LOW();
  I2C_SCL_HIGH();
  I2C_SDA_HIGH();
}

// I2C receive one data byte from the slave (ack=0 for last byte, ack>0 if more bytes to follow)
uint8_t I2C_read(uint8_t ack) {
  uint8_t data = 0;
  I2C_SDA_HIGH();
  for(uint8_t i=8; i; i--) {
    data <<= 1;
    I2C_SCL_HIGH();
    if(I2C_SDA_READ()) data |= 1;
    I2C_SCL_LOW();
  }
  if(ack) I2C_SDA_LOW();
  I2C_CLOCKOUT();
  return data;
}

// ===================================================================================
// RDA5807 Implementation
// ===================================================================================

// RDA definitions
#define RDA_ADDR_SEQ    0x20                      // RDA I2C write address for sequential access
#define RDA_ADDR_INDEX  0x22                      // RDA I2C write address for indexed access
#define RDA_VOL         1                         // start volume

// RDA register definitions
enum{ RDA_REG_2, RDA_REG_3, RDA_REG_4, RDA_REG_5, RDA_REG_6, RDA_REG_7 };
enum{ RDA_REG_A, RDA_REG_B, RDA_REG_C, RDA_REG_D, RDA_REG_E, RDA_REG_F };
uint16_t RDA_read_regs[6];
uint16_t RDA_write_regs[6] = {
  0b1101001000001101,
  0b0001010111000000,
  0b0000101000000000,
  0b1000100010000000,
  0b0000000000000000,
  0b0000000000000000
};

// RDA state macros
#define RDA_isTuning    (~RDA_read_regs[RDA_REG_A] & 0x4000)
#define RDA_channel     (RDA_read_regs[RDA_REG_A] & 0x03FF)
#define RDA_isTunedToChannel  (RDA_read_regs[RDA_REG_B] & 0x0100)

// RDA write specified register
void RDA_writeReg(uint8_t reg) {
  I2C_start(RDA_ADDR_INDEX);
  I2C_write(0x02 + reg);
  I2C_write(RDA_write_regs[reg] >> 8);
  I2C_write(RDA_write_regs[reg]);
  I2C_stop();
}

// RDA write all registers
void RDA_writeAllRegs(void) {
  I2C_start(RDA_ADDR_SEQ);
  for(uint8_t i=0; i<6; i++) {
    I2C_write(RDA_write_regs[i] >> 8);
    I2C_write(RDA_write_regs[i]);
  }
  I2C_stop();
}

// RDA read all registers
void RDA_readAllRegs(void) {
  I2C_start(RDA_ADDR_SEQ | 1);                    // start I2C for sequential read from RDA
  for(uint8_t i=0; i<6; i++) {                    // read 6 registers
    RDA_read_regs[i] = (uint16_t)(I2C_read(1) << 8) | I2C_read(i == 5 ? 0 : 1);
  }
  I2C_stop();                                     // stop I2C
}

// RDA initialize tuner
void RDA_init(void) {
  RDA_write_regs[RDA_REG_2] |=  0x0002;
  RDA_write_regs[RDA_REG_5] |=  RDA_VOL;
  RDA_writeAllRegs();
  RDA_write_regs[RDA_REG_2] &= ~0x0002;
  RDA_writeReg(RDA_REG_2);
}

// RDA set volume
void RDA_setVolume(uint8_t vol) {
  RDA_write_regs[RDA_REG_5] &= ~0x000F;
  RDA_write_regs[RDA_REG_5] |=  vol;
  RDA_writeReg(RDA_REG_5);
}

// RDA tune to a specified channel
void RDA_setChannel(uint16_t chan) {
  RDA_write_regs[RDA_REG_3] &= ~0xFFC0;
  RDA_write_regs[RDA_REG_3] |= (chan << 6) | 0x0010;
  RDA_writeReg(RDA_REG_3);
}

// RDA seek next channel
void RDA_seekUp(void) {
  RDA_write_regs[RDA_REG_2] |=  0x0100;
  RDA_writeReg(RDA_REG_2);
}

// RDA seek previous channel
void RDA_seekDown(void) {
  RDA_write_regs[RDA_REG_2] |=  0x0200;
  RDA_writeReg(RDA_REG_2);
}

// Wait until tuning completed
void RDA_waitTuning(void) {
  do {
    _delay_ms(100);
    RDA_readAllRegs();
  } while(RDA_isTuning);
}

// ===================================================================================
// EEPROM Functions
// ===================================================================================

// Update frequency and volume stored in EEPROM
void EEPROM_update() {
  eeprom_update_word((uint16_t*)0, EEPROM_IDENT);
  eeprom_update_word((uint16_t*)2, channel);
  eeprom_update_byte((uint8_t*)4, volume);
}

// Read frequency and volume stored in EEPROM
uint8_t EEPROM_get() {
  uint16_t identifier = eeprom_read_word((const uint16_t*)0);
  if (identifier == EEPROM_IDENT) {
    channel = eeprom_read_word((const uint16_t*)2);
    volume  = eeprom_read_byte((const uint8_t*)4);
    return 1;
  }
  return 0;
}

// ===================================================================================
// Main Function
// ===================================================================================

int main(void) {
  // Setup
  _PROTECTED_WRITE(CLKCTRL.MCLKCTRLB, 7);         // set clock frequency to 1 MHz
  pinPullup(PIN_SWITCH);                          // enable pullup for switch pin

  // Start the tuner
  RDA_init();
  if(EEPROM_get()) {
    RDA_setChannel(channel - 2);
    RDA_waitTuning();
  }
  RDA_setVolume(volume);

  // Loop
  while(1) {
    uint16_t pressTime = 0;
    while(!pinRead(PIN_SWITCH)) {                 // Button pressed
      _delay_ms(10);
      pressTime += 10;
      if(pressTime == 2000) {                     // 2s hold: Seek down
        RDA_seekDown();
        RDA_waitTuning();
        channel = RDA_channel;
        EEPROM_update();
      }
      if(pressTime == 10000) {                    // 10s hold: Increase volume
        if(volume < 15) volume++;
        RDA_setVolume(volume);
        EEPROM_update();
      }
      if(pressTime == 30000) {                    // 30s hold: Decrease volume
        if(volume > 0) volume--;
        RDA_setVolume(volume);
        EEPROM_update();
      }
    }
    if(pressTime > 0 && pressTime < 2000) {       // Quick press (<2s): Seek up
      RDA_seekUp();
      RDA_waitTuning();
      channel = RDA_channel;
      EEPROM_update();
    }
  }
}