/*
 * Flappy Bird Game for ESP32 with ST7789 240x320 Display using TFT_eSPI
 * Adapted from the original Arduino project at:
 * https://howtomechatronics.com/projects/arduino-game-project-replica-of-flappy-bird-for-arduino-on-a-tft-touch-screen/
 * 
 * This version uses a button for jumping instead of touch input.
 * Connect a button to GPIO 0 (BOOT button on many ESP32 boards) and GND.
 * The display is set to landscape mode (320x240 logical resolution).
 * 
 * Install TFT_eSPI library via Arduino Library Manager.
 * Configure TFT_eSPI for ST7789 in User_Setup_Select.h (e.g., include Setup43_ST7789_ESP32.h or similar).
 * Adjust pins in the setup file as per your hardware.
 */

#include <TFT_eSPI.h>

TFT_eSPI tft = TFT_eSPI();

const int buttonPin = 0;  // Button pin for jumping (use INPUT_PULLUP)

int xP = 320;  // Pillar x position
int yP = 100;  // Pillar gap y position (bottom of upper pillar)
int yB = 50;   // Bird y position
int movingRate = 3;  // Pillar movement speed
float fallRate = 0;
int fallRateInt = 0;
int score = 0;
int lastSpeedUpScore = 0;
int highestScore = 0;  // Simple variable, no EEPROM
boolean gameStarted = false;
boolean buttonPressed = false;  // For debounce

// Colors in RGB565 format
uint16_t skyColor = tft.color565(114, 198, 206);
uint16_t groundColor = tft.color565(221, 216, 148);
uint16_t grassColor = tft.color565(47, 175, 68);
uint16_t pillarColor = tft.color565(0, 175, 0);
uint16_t pillarBorderColor = TFT_BLACK;
uint16_t birdColor = TFT_YELLOW;
uint16_t textColor = TFT_BLACK;

void setup() {
  pinMode(buttonPin, INPUT_PULLUP);
  randomSeed(analogRead(0));  // For random pillar positions

  tft.init();
  tft.setRotation(1);  // Landscape mode: 320 width x 240 height
  tft.fillScreen(skyColor);

  initiateGame();
}

void loop() {
  if (!gameStarted) return;

  // Redraw the entire screen each frame to simplify drawing (ESP32 is fast enough)
  tft.fillScreen(skyColor);

  // Draw ground
  tft.fillRect(0, 215, 320, 25, groundColor);
  tft.fillRect(0, 205, 320, 10, grassColor);

  // Draw pillars
  tft.fillRect(xP, 0, 50, yP, pillarColor);  // Upper pillar
  tft.fillRect(xP, yP + 60, 50, 205 - (yP + 60) + 1, pillarColor);  // Lower pillar
  // Borders for pillars (optional)
  tft.drawRect(xP, 0, 50, yP, pillarBorderColor);
  tft.drawRect(xP, yP + 60, 50, 205 - (yP + 60) + 1, pillarBorderColor);

  // Draw bird (simple circle for simplicity; could be replaced with a bitmap)
  tft.fillCircle(50, yB, 10, birdColor);

  // Draw score
  tft.setTextColor(textColor);
  tft.setTextSize(2);
  tft.setCursor(5, 220);
  tft.print("Score: ");
  tft.print(score);

  // Move pillars
  xP -= movingRate;
  if (xP <= -51) {
    xP = 320;
    yP = random(20, 120);  // Random gap position
    score++;
    if ((score - lastSpeedUpScore) == 5) {
      movingRate++;
      lastSpeedUpScore = score;
    }
  }

  // Bird physics
  fallRate += 0.4;
  fallRateInt = int(fallRate);
  yB += fallRateInt;

  // Button input for jump
  if (digitalRead(buttonPin) == LOW && !buttonPressed) {
    fallRate = -6;  // Jump
    buttonPressed = true;
  } else if (digitalRead(buttonPin) == HIGH && buttonPressed) {
    buttonPressed = false;
  }

  // Check collisions
  if (yB + 10 >= 205 || yB - 10 <= 0) {  // Hit ground or ceiling
    gameOver();
  }
  if (xP <= 60 && xP >= -10) {  // Bird overlapping pillar x-range
    if (yB - 10 <= yP || yB + 10 >= yP + 60) {  // Hit pillar
      gameOver();
    }
  }

  delay(20);  // Control frame rate (~50 FPS)
}

// Initiate or restart the game
void initiateGame() {
  tft.fillScreen(skyColor);
  tft.fillRect(0, 215, 320, 25, groundColor);
  tft.fillRect(0, 205, 320, 10, grassColor);

  tft.setTextColor(textColor);
  tft.setTextSize(2);
  tft.setCursor(5, 220);
  tft.print("Score:");

  tft.setCursor(60, 100);
  tft.print("Press button");
  tft.setCursor(60, 120);
  tft.print("to start");

  // Wait for button press
  while (digitalRead(buttonPin) == HIGH) {
    delay(100);
  }

  // Clear start text
  tft.fillRect(60, 100, 200, 40, skyColor);
  tft.fillRect(60, 120, 200, 40, skyColor);

  // Reset game variables
  gameStarted = true;
  yB = 100;
  fallRate = 0;
  score = 0;
  movingRate = 3;
  lastSpeedUpScore = 0;
  xP = 320;
  yP = random(20, 120);
}

// Game over screen
void gameOver() {
  tft.setTextSize(3);
  tft.setCursor(80, 80);
  tft.print("Game Over");

  tft.setTextSize(2);
  tft.setCursor(100, 120);
  tft.print("Score: ");
  tft.print(score);

  if (score > highestScore) {
    highestScore = score;
  }
  tft.setCursor(100, 140);
  tft.print("High: ");
  tft.print(highestScore);

  tft.setCursor(60, 170);
  tft.print("Press button");
  tft.setCursor(60, 190);
  tft.print("to restart");

  // Wait for button press
  while (digitalRead(buttonPin) == HIGH) {
    delay(100);
  }

  initiateGame();
}