#include <Wire.h>
#include <MiCS6814-I2C.h>
#include "bme69x.h"
#include <WiFi.h>
#include <WebServer.h>

// I2C address for BME690
#define BME69X_I2C_ADDR BME69X_I2C_ADDR_LOW // 0x76, use HIGH for 0x77

MiCS6814 sensor;
bool sensorConnected;

struct bme69x_dev dev;
struct bme69x_conf conf;
struct bme69x_heatr_conf heatr_conf;
struct bme69x_data data;

WebServer server(80);

// Global sensor values
float co = 0.0, no2 = 0.0, nh3 = 0.0, c3h8 = 0.0, c4h10 = 0.0, ch4 = 0.0, h2 = 0.0, c2h5oh = 0.0;
float temp = 0.0, press = 0.0, hum = 0.0, gas_res = 0.0;
float aqi = 0.0;

int8_t i2c_read(uint8_t reg_addr, uint8_t *reg_data, uint32_t len, void *intf_ptr) {
  Wire.beginTransmission(*(uint8_t*)intf_ptr);
  Wire.write(reg_addr);
  Wire.endTransmission();
  Wire.requestFrom(*(uint8_t*)intf_ptr, (uint8_t)len);
  for (uint32_t i = 0; i < len; i++) {
    reg_data[i] = Wire.read();
  }
  return 0;
}

int8_t i2c_write(uint8_t reg_addr, const uint8_t *reg_data, uint32_t len, void *intf_ptr) {
  Wire.beginTransmission(*(uint8_t*)intf_ptr);
  Wire.write(reg_addr);
  for (uint32_t i = 0; i < len; i++) {
    Wire.write(reg_data[i]);
  }
  Wire.endTransmission();
  return 0;
}

void delay_us(uint32_t period, void *intf_ptr) {
  delayMicroseconds(period);
}

void handleRoot() {
  String html = R"(
    <!DOCTYPE html>
    <html>
    <head>
      <title>Indus Environment / Gas Sensor</title>
      <style>
        body { background-color: white; font-family: Arial, sans-serif; margin: 0; padding: 20px; }
        h1 { text-align: center; color: #333; }
        .section { margin-bottom: 40px; }
        .cards-container { display: flex; flex-wrap: wrap; justify-content: center; }
        .card { background-color: #f0f0f0; box-shadow: 0 4px 8px 0 rgba(0,0,0,0.2), 0 6px 20px 0 rgba(0,0,0,0.19); 
                padding: 20px; margin: 10px; border-radius: 10px; width: 200px; text-align: center; 
                transition: transform 0.3s; }
        .card:hover { transform: translateY(-5px); }
        .card-title { font-size: 18px; color: #555; margin-bottom: 10px; }
        .card-value { font-size: 24px; font-weight: bold; color: #222; }
        .card-unit { font-size: 16px; color: #777; }
        .list-container { max-width: 600px; margin: 0 auto; }
        ul { list-style-type: none; padding: 0; }
        li { background-color: #f8f8f8; margin: 5px 0; padding: 10px; border-radius: 5px; 
             box-shadow: 0 2px 4px rgba(0,0,0,0.1); display: flex; justify-content: space-between; }
        .list-label { font-weight: bold; color: #333; }
        .list-value { color: #555; }
      </style>
      <script>
        function updateData() {
          fetch('/data')
            .then(response => response.json())
            .then(data => {
              // Update cards and list
              document.getElementById('temp-value').innerText = data.temp.toFixed(2);
              document.getElementById('hum-value').innerText = data.hum.toFixed(2);
              document.getElementById('press-value').innerText = (data.press / 100.0).toFixed(2);
              document.getElementById('aqi-value').innerText = data.aqi.toFixed(0);
              document.getElementById('gasres-value').innerText = data.gas_res.toFixed(0);
              document.getElementById('co-value').innerText = data.co.toFixed(2);
              document.getElementById('no2-value').innerText = data.no2.toFixed(2);
              document.getElementById('nh3-value').innerText = data.nh3.toFixed(2);
              document.getElementById('c3h8-value').innerText = data.c3h8.toFixed(2);
              document.getElementById('c4h10-value').innerText = data.c4h10.toFixed(2);
              document.getElementById('ch4-value').innerText = data.ch4.toFixed(2);
              document.getElementById('h2-value').innerText = data.h2.toFixed(2);
              document.getElementById('c2h5oh-value').innerText = data.c2h5oh.toFixed(2);

              // Update list
              document.getElementById('list-temp').innerText = data.temp.toFixed(2) + ' °C';
              document.getElementById('list-hum').innerText = data.hum.toFixed(2) + ' %';
              document.getElementById('list-press').innerText = (data.press / 100.0).toFixed(2) + ' hPa';
              document.getElementById('list-aqi').innerText = data.aqi.toFixed(0);
              document.getElementById('list-gasres').innerText = data.gas_res.toFixed(0) + ' ohms';
              document.getElementById('list-co').innerText = data.co.toFixed(2) + ' ppm';
              document.getElementById('list-no2').innerText = data.no2.toFixed(2) + ' ppm';
              document.getElementById('list-nh3').innerText = data.nh3.toFixed(2) + ' ppm';
              document.getElementById('list-c3h8').innerText = data.c3h8.toFixed(2) + ' ppm';
              document.getElementById('list-c4h10').innerText = data.c4h10.toFixed(2) + ' ppm';
              document.getElementById('list-ch4').innerText = data.ch4.toFixed(2) + ' ppm';
              document.getElementById('list-h2').innerText = data.h2.toFixed(2) + ' ppm';
              document.getElementById('list-c2h5oh').innerText = data.c2h5oh.toFixed(2) + ' ppm';
            });
        }
        setInterval(updateData, 1000);
        window.onload = updateData;
      </script>
    </head>
    <body>
      <h1>Indus Environment / Gas Sensor</h1>
      
      <div class="section">
        <h2>Dashboard (Cards)</h2>
        <div class="cards-container">
          <div class="card">
            <div class="card-title">Temperature</div>
            <div class="card-value"><span id="temp-value">-</span></div>
            <div class="card-unit">°C</div>
          </div>
          <div class="card">
            <div class="card-title">Humidity</div>
            <div class="card-value"><span id="hum-value">-</span></div>
            <div class="card-unit">%</div>
          </div>
          <div class="card">
            <div class="card-title">Pressure</div>
            <div class="card-value"><span id="press-value">-</span></div>
            <div class="card-unit">hPa</div>
          </div>
          <div class="card">
            <div class="card-title">AQI</div>
            <div class="card-value"><span id="aqi-value">-</span></div>
            <div class="card-unit"></div>
          </div>
          <div class="card">
            <div class="card-title">Gas Resistance</div>
            <div class="card-value"><span id="gasres-value">-</span></div>
            <div class="card-unit">ohms</div>
          </div>
          <div class="card">
            <div class="card-title">CO</div>
            <div class="card-value"><span id="co-value">-</span></div>
            <div class="card-unit">ppm</div>
          </div>
          <div class="card">
            <div class="card-title">NO2</div>
            <div class="card-value"><span id="no2-value">-</span></div>
            <div class="card-unit">ppm</div>
          </div>
          <div class="card">
            <div class="card-title">NH3</div>
            <div class="card-value"><span id="nh3-value">-</span></div>
            <div class="card-unit">ppm</div>
          </div>
          <div class="card">
            <div class="card-title">C3H8</div>
            <div class="card-value"><span id="c3h8-value">-</span></div>
            <div class="card-unit">ppm</div>
          </div>
          <div class="card">
            <div class="card-title">C4H10</div>
            <div class="card-value"><span id="c4h10-value">-</span></div>
            <div class="card-unit">ppm</div>
          </div>
          <div class="card">
            <div class="card-title">CH4</div>
            <div class="card-value"><span id="ch4-value">-</span></div>
            <div class="card-unit">ppm</div>
          </div>
          <div class="card">
            <div class="card-title">H2</div>
            <div class="card-value"><span id="h2-value">-</span></div>
            <div class="card-unit">ppm</div>
          </div>
          <div class="card">
            <div class="card-title">C2H5OH</div>
            <div class="card-value"><span id="c2h5oh-value">-</span></div>
            <div class="card-unit">ppm</div>
          </div>
        </div>
      </div>
      
      <div class="section">
        <h2>Data List</h2>
        <div class="list-container">
          <ul>
            <li><span class="list-label">Temperature:</span> <span class="list-value" id="list-temp">-</span></li>
            <li><span class="list-label">Humidity:</span> <span class="list-value" id="list-hum">-</span></li>
            <li><span class="list-label">Pressure:</span> <span class="list-value" id="list-press">-</span></li>
            <li><span class="list-label">AQI:</span> <span class="list-value" id="list-aqi">-</span></li>
            <li><span class="list-label">Gas Resistance:</span> <span class="list-value" id="list-gasres">-</span></li>
            <li><span class="list-label">CO:</span> <span class="list-value" id="list-co">-</span></li>
            <li><span class="list-label">NO2:</span> <span class="list-value" id="list-no2">-</span></li>
            <li><span class="list-label">NH3:</span> <span class="list-value" id="list-nh3">-</span></li>
            <li><span class="list-label">C3H8:</span> <span class="list-value" id="list-c3h8">-</span></li>
            <li><span class="list-label">C4H10:</span> <span class="list-value" id="list-c4h10">-</span></li>
            <li><span class="list-label">CH4:</span> <span class="list-value" id="list-ch4">-</span></li>
            <li><span class="list-label">H2:</span> <span class="list-value" id="list-h2">-</span></li>
            <li><span class="list-label">C2H5OH:</span> <span class="list-value" id="list-c2h5oh">-</span></li>
          </ul>
        </div>
      </div>
    </body>
    </html>
  )";
  server.send(200, "text/html", html);
}

void handleData() {
  String json = "{";
  json += "\"temp\":" + String(temp) + ",";
  json += "\"press\":" + String(press) + ",";
  json += "\"hum\":" + String(hum) + ",";
  json += "\"gas_res\":" + String(gas_res) + ",";
  json += "\"aqi\":" + String(aqi) + ",";
  json += "\"co\":" + String(co) + ",";
  json += "\"no2\":" + String(no2) + ",";
  json += "\"nh3\":" + String(nh3) + ",";
  json += "\"c3h8\":" + String(c3h8) + ",";
  json += "\"c4h10\":" + String(c4h10) + ",";
  json += "\"ch4\":" + String(ch4) + ",";
  json += "\"h2\":" + String(h2) + ",";
  json += "\"c2h5oh\":" + String(c2h5oh);
  json += "}";
  server.send(200, "application/json", json);
}

void setup() {
  // Initialize serial connection
  Serial.begin(115200);
  while (!Serial);

  // Initialize I2C
  Wire.begin();

  // Connect to MiCS-6814 sensor using default I2C address (0x04)
  sensorConnected = sensor.begin();
  if (sensorConnected) {
    Serial.println("Connected to MiCS-6814 sensor");
    sensor.powerOn();
  } else {
    Serial.println("Couldn't connect to MiCS-6814 sensor");
  }

  // Initialize BME690
  uint8_t dev_addr = BME69X_I2C_ADDR;
  dev.intf_ptr = &dev_addr;
  dev.intf = BME69X_I2C_INTF;
  dev.read = i2c_read;
  dev.write = i2c_write;
  dev.delay_us = delay_us;

  if (bme69x_init(&dev) != BME69X_OK) {
    Serial.println("BME690 init failed!");
    while (1);
  }

  // Set up BME690 oversampling and filter
  conf.filter = BME69X_FILTER_OFF;
  conf.odr = BME69X_ODR_NONE;
  conf.os_hum = BME69X_OS_16X;
  conf.os_pres = BME69X_OS_1X;
  conf.os_temp = BME69X_OS_2X;
  bme69x_set_conf(&conf, &dev);

  // BME690 heater configuration
  heatr_conf.enable = BME69X_ENABLE;
  heatr_conf.heatr_temp = 300; // 300 °C
  heatr_conf.heatr_dur = 100;  // 100 ms
  bme69x_set_heatr_conf(BME69X_FORCED_MODE, &heatr_conf, &dev);

  // Set up WiFi AP
  const char* ssid = "IndusSensorAP";
  const char* password = "password123";
  WiFi.softAP(ssid, password);
  Serial.print("AP IP address: ");
  Serial.println(WiFi.softAPIP());

  // Set up web server
  server.on("/", handleRoot);
  server.on("/data", handleData);
  server.begin();
  Serial.println("Web server started");
}

void loop() {
  // Handle web clients
  server.handleClient();

  // Read MiCS-6814 data
  if (sensorConnected) {
    co = sensor.measureCO();
    no2 = sensor.measureNO2();
    nh3 = sensor.measureNH3();
    c3h8 = sensor.measureC3H8();
    c4h10 = sensor.measureC4H10();
    ch4 = sensor.measureCH4();
    h2 = sensor.measureH2();
    c2h5oh = sensor.measureC2H5OH();
  } else {
    co = no2 = nh3 = c3h8 = c4h10 = ch4 = h2 = c2h5oh = -1.0;
  }

  // Trigger BME690 measurement in forced mode
  bme69x_set_op_mode(BME69X_FORCED_MODE, &dev);
  dev.delay_us(100000, dev.intf_ptr); // Wait 100 ms

  // Read BME690 data
  uint8_t n_fields;
  if (bme69x_get_data(BME69X_FORCED_MODE, &data, &n_fields, &dev) == BME69X_OK && n_fields > 0) {
    temp = data.temperature;
    press = data.pressure;
    hum = data.humidity;
    gas_res = data.gas_resistance;

    // Calculate AQI roughly
    float gas_log = log(gas_res);
    aqi = (gas_log - log(1000000.0)) / (log(1000.0) - log(1000000.0)) * 100.0;
    if (aqi < 0) aqi = 0;
    if (aqi > 100) aqi = 100;
  } else {
    temp = press = hum = gas_res = aqi = -1.0;
  }

  delay(1000); // Wait 1 second before next reading
}