#include <SPI.h>
#include <TFT_eSPI.h>
#include <Wire.h>
#include <MiCS6814-I2C.h>
#include "bme69x.h"
#include "Free_Fonts.h" // Include the header file for free fonts
#include <math.h>

#define PI 3.14159265

// I2C address for BME690
#define BME69X_I2C_ADDR BME69X_I2C_ADDR_LOW // 0x76, use HIGH for 0x77

TFT_eSPI tft = TFT_eSPI();

MiCS6814 sensor;
bool sensorConnected;

struct bme69x_dev dev;
struct bme69x_conf conf;
struct bme69x_heatr_conf heatr_conf;
struct bme69x_data data;

int8_t i2c_read(uint8_t reg_addr, uint8_t *reg_data, uint32_t len, void *intf_ptr) {
  Wire.beginTransmission(*(uint8_t*)intf_ptr);
  Wire.write(reg_addr);
  Wire.endTransmission();
  Wire.requestFrom(*(uint8_t*)intf_ptr, (uint8_t)len);
  for (uint32_t i = 0; i < len; i++) {
    reg_data[i] = Wire.read();
  }
  return 0;
}

int8_t i2c_write(uint8_t reg_addr, const uint8_t *reg_data, uint32_t len, void *intf_ptr) {
  Wire.beginTransmission(*(uint8_t*)intf_ptr);
  Wire.write(reg_addr);
  for (uint32_t i = 0; i < len; i++) {
    Wire.write(reg_data[i]);
  }
  Wire.endTransmission();
  return 0;
}

void delay_us(uint32_t period, void *intf_ptr) {
  delayMicroseconds(period);
}

// Function to draw a filled arc (ring segment) by drawing radial lines
void fillArc(int cx, int cy, int start_angle, int end_angle, int outer_r, int inner_r, uint16_t color) {
  for (int angle = start_angle; angle <= end_angle; angle++) {
    float rad = (float)angle * PI / 180.0;
    int x1 = cx + (int)(inner_r * cos(rad));
    int y1 = cy + (int)(inner_r * sin(rad));
    int x2 = cx + (int)(outer_r * cos(rad));
    int y2 = cy + (int)(outer_r * sin(rad));
    tft.drawLine(x1, y1, x2, y2, color);
  }
}

void drawENVScreen() {
  tft.fillScreen(TFT_BLACK);

  // Connection status dot (default green)
  tft.fillCircle(20, 20, 10, TFT_GREEN);

  // Calculate AQI roughly (assuming higher gas resistance = better air quality, AQI 0-100 where 0 is good)
  float gas_log = log(data.gas_resistance);
  float aqi = (gas_log - log(1000000.0)) / (log(1000.0) - log(1000000.0)) * 100.0;
  if (aqi < 0) aqi = 0;
  if (aqi > 100) aqi = 100;
  float aqi_percent = aqi / 100.0;

  // Temp gauge (left, red/orange tones)
  int cx1 = 80, cy = 120, outer = 60, inner = 45;
  float temp_percent = data.temperature / 50.0; // Assume 0-50°C range
  if (temp_percent > 1) temp_percent = 1;
  if (temp_percent < 0) temp_percent = 0;
  uint16_t gray = tft.color565(50, 50, 50);
  int start_ang = 180; // Start at left
  int sweep = 180; // Semi-circle to right
  fillArc(cx1, cy, start_ang, start_ang + sweep, outer, inner, gray); // Background
  fillArc(cx1, cy, start_ang, start_ang + (int)(sweep * temp_percent), outer, inner, TFT_ORANGE); // Fill
  tft.setTextDatum(MC_DATUM);
  tft.setFreeFont(FF24); // Large font for value
  tft.setTextColor(TFT_YELLOW);
  tft.drawFloat(data.temperature, 1, cx1, cy - 10, GFXFF);
  tft.setFreeFont(FF18); // Smaller for unit
  tft.drawString("°C", cx1 + 20, cy - 10, GFXFF);
  tft.setFreeFont(FF17);
  tft.setTextColor(TFT_WHITE);
  tft.drawString("Temp", cx1, cy + outer + 10, GFXFF);

  // Humidity gauge (middle, blue tones)
  int cx2 = 160;
  float hum_percent = data.humidity / 100.0; // 0-100%
  fillArc(cx2, cy, start_ang, start_ang + sweep, outer, inner, gray);
  fillArc(cx2, cy, start_ang, start_ang + (int)(sweep * hum_percent), outer, inner, TFT_CYAN);
  tft.setFreeFont(FF24);
  tft.setTextColor(TFT_LIGHTGREY);
  tft.drawFloat(data.humidity, 1, cx2, cy - 10, GFXFF);
  tft.setFreeFont(FF18);
  tft.drawString("%", cx2 + 20, cy - 10, GFXFF);
  tft.setFreeFont(FF17);
  tft.setTextColor(TFT_WHITE);
  tft.drawString("Humidity", cx2, cy + outer + 10, GFXFF);

  // AQI gauge (right, green to red)
  int cx3 = 240;
  uint16_t aqi_color = (aqi_percent < 0.5) ? TFT_GREEN : TFT_RED;
  fillArc(cx3, cy, start_ang, start_ang + sweep, outer, inner, gray);
  fillArc(cx3, cy, start_ang, start_ang + (int)(sweep * (1.0 - aqi_percent)), outer, inner, aqi_color); // Invert for good low
  tft.setFreeFont(FF24);
  tft.setTextColor(TFT_WHITE);
  tft.drawFloat(aqi, 0, cx3, cy - 10, GFXFF);
  tft.setFreeFont(FF17);
  tft.setTextColor(TFT_WHITE);
  tft.drawString("AQI", cx3, cy + outer + 10, GFXFF);

  // Text for pressure and gas resistance at bottom
  tft.setFreeFont(FF19);
  tft.setTextColor(TFT_MAGENTA);
  tft.setTextDatum(TL_DATUM);
  tft.drawString("Pressure:", 20, 210, GFXFF);
  tft.setTextColor(TFT_WHITE);
  tft.drawFloat(data.pressure / 100.0, 2, 120, 210, GFXFF);
  tft.drawString("hPa", 200, 210, GFXFF);

  tft.setTextColor(TFT_MAGENTA);
  tft.drawString("Gas Res:", 20, 225, GFXFF);
  tft.setTextColor(TFT_WHITE);
  tft.drawFloat(data.gas_resistance, 0, 120, 225, GFXFF);
  tft.drawString("ohms", 200, 225, GFXFF);
}

void drawGasScreen() {
  tft.fillScreen(TFT_BLACK);

  // Display gases in a list with varying colors and fonts for beauty
  int y_pos = 20;
 // int label_font = FF18; // Smaller for labels
 // int value_font = FF22; // Larger for values

  // CO
  tft.setFreeFont(FF18);
  tft.setTextColor(TFT_RED);
  tft.setTextDatum(TL_DATUM);
  tft.drawString("CO:", 20, y_pos, GFXFF);
  tft.setFreeFont(FF22);
  tft.setTextColor(TFT_WHITE);
  if (sensorConnected) {
    tft.drawFloat(sensor.measureCO(), 2, 100, y_pos, GFXFF);
  } else {
    tft.drawString("-", 100, y_pos, GFXFF);
  }
  tft.setFreeFont(FF18);
  tft.drawString("ppm", 200, y_pos, GFXFF);
  y_pos += 25;

  // NO2
  tft.setFreeFont(FF18);
  tft.setTextColor(TFT_ORANGE);
  tft.drawString("NO2:", 20, y_pos, GFXFF);
  tft.setFreeFont(FF22);
  tft.setTextColor(TFT_WHITE);
  if (sensorConnected) {
    tft.drawFloat(sensor.measureNO2(), 2, 100, y_pos, GFXFF);
  } else {
    tft.drawString("-", 100, y_pos, GFXFF);
  }
  tft.setFreeFont(FF18);
  tft.drawString("ppm", 200, y_pos, GFXFF);
  y_pos += 25;

  // NH3
  tft.setFreeFont(FF18);
  tft.setTextColor(TFT_YELLOW);
  tft.drawString("NH3:", 20, y_pos, GFXFF);
  tft.setFreeFont(FF22);
  tft.setTextColor(TFT_WHITE);
  if (sensorConnected) {
    tft.drawFloat(sensor.measureNH3(), 2, 100, y_pos, GFXFF);
  } else {
    tft.drawString("-", 100, y_pos, GFXFF);
  }
  tft.setFreeFont(FF18);
  tft.drawString("ppm", 200, y_pos, GFXFF);
  y_pos += 25;

  // C3H8
  tft.setFreeFont(FF18);
  tft.setTextColor(TFT_GREEN);
  tft.drawString("C3H8:", 20, y_pos, GFXFF);
  tft.setFreeFont(FF22);
  tft.setTextColor(TFT_WHITE);
  if (sensorConnected) {
    tft.drawFloat(sensor.measureC3H8(), 2, 100, y_pos, GFXFF);
  } else {
    tft.drawString("-", 100, y_pos, GFXFF);
  }
  tft.setFreeFont(FF18);
  tft.drawString("ppm", 200, y_pos, GFXFF);
  y_pos += 25;

  // C4H10
  tft.setFreeFont(FF18);
  tft.setTextColor(TFT_CYAN);
  tft.drawString("C4H10:", 20, y_pos, GFXFF);
  tft.setFreeFont(FF22);
  tft.setTextColor(TFT_WHITE);
  if (sensorConnected) {
    tft.drawFloat(sensor.measureC4H10(), 2, 100, y_pos, GFXFF);
  } else {
    tft.drawString("-", 100, y_pos, GFXFF);
  }
  tft.setFreeFont(FF18);
  tft.drawString("ppm", 200, y_pos, GFXFF);
  y_pos += 25;

  // CH4
  tft.setFreeFont(FF18);
  tft.setTextColor(TFT_BLUE);
  tft.drawString("CH4:", 20, y_pos, GFXFF);
  tft.setFreeFont(FF22);
  tft.setTextColor(TFT_WHITE);
  if (sensorConnected) {
    tft.drawFloat(sensor.measureCH4(), 2, 100, y_pos, GFXFF);
  } else {
    tft.drawString("-", 100, y_pos, GFXFF);
  }
  tft.setFreeFont(FF18);
  tft.drawString("ppm", 200, y_pos, GFXFF);
  y_pos += 25;

  // H2
  tft.setFreeFont(FF18);
  tft.setTextColor(TFT_MAGENTA);
  tft.drawString("H2:", 20, y_pos, GFXFF);
  tft.setFreeFont(FF22);
  tft.setTextColor(TFT_WHITE);
  if (sensorConnected) {
    tft.drawFloat(sensor.measureH2(), 2, 100, y_pos, GFXFF);
  } else {
    tft.drawString("-", 100, y_pos, GFXFF);
  }
  tft.setFreeFont(FF18);
  tft.drawString("ppm", 200, y_pos, GFXFF);
  y_pos += 25;

  // C2H5OH
  tft.setFreeFont(FF18);
  tft.setTextColor(TFT_WHITE);
  tft.drawString("C2H5OH:", 20, y_pos, GFXFF);
  tft.setFreeFont(FF22);
  tft.setTextColor(TFT_WHITE);
  if (sensorConnected) {
    tft.drawFloat(sensor.measureC2H5OH(), 2, 100, y_pos, GFXFF);
  } else {
    tft.drawString("-", 100, y_pos, GFXFF);
  }
  tft.setFreeFont(FF18);
  tft.drawString("ppm", 200, y_pos, GFXFF);
}

void setup() {
  // Initialize serial connection
  Serial.begin(115200);
  while (!Serial);

  // Initialize TFT
  tft.init();
  tft.setRotation(1); // Landscape mode, 320x240
  tft.fillScreen(TFT_BLACK);
  tft.setTextColor(TFT_WHITE, TFT_BLACK);

  // Initialize I2C
  Wire.begin();

  // Connect to MiCS-6814 sensor using default I2C address (0x04)
  sensorConnected = sensor.begin();
  if (sensorConnected) {
    Serial.println("Connected to MiCS-6814 sensor");
    sensor.powerOn();
  } else {
    Serial.println("Couldn't connect to MiCS-6814 sensor");
  }

  // Initialize BME690
  uint8_t dev_addr = BME69X_I2C_ADDR;
  dev.intf_ptr = &dev_addr;
  dev.intf = BME69X_I2C_INTF;
  dev.read = i2c_read;
  dev.write = i2c_write;
  dev.delay_us = delay_us;

  if (bme69x_init(&dev) != BME69X_OK) {
    Serial.println("BME690 init failed!");
    while (1);
  }

  // Set up BME690 oversampling and filter
  conf.filter = BME69X_FILTER_OFF;
  conf.odr = BME69X_ODR_NONE;
  conf.os_hum = BME69X_OS_16X;
  conf.os_pres = BME69X_OS_1X;
  conf.os_temp = BME69X_OS_2X;
  bme69x_set_conf(&conf, &dev);

  // BME690 heater configuration
  heatr_conf.enable = BME69X_ENABLE;
  heatr_conf.heatr_temp = 300; // 300 °C
  heatr_conf.heatr_dur = 100;  // 100 ms
  bme69x_set_heatr_conf(BME69X_FORCED_MODE, &heatr_conf, &dev);
}

void loop() {
  // Read MiCS-6814 data (done in draw if needed, but since measure is called directly)

  // Trigger BME690 measurement in forced mode
  bme69x_set_op_mode(BME69X_FORCED_MODE, &dev);
  dev.delay_us(100000, dev.intf_ptr); // Wait 100 ms

  // Read BME690 data
  uint8_t n_fields = 0;
  bme69x_get_data(BME69X_FORCED_MODE, &data, &n_fields, &dev);

  // Determine screen based on time: first 5 seconds ENV, next 40 seconds Gas, repeat every 45 seconds
  unsigned long timePassed = millis();
  unsigned long cycle = timePassed % 45000;
  if (cycle < 5000) {
    drawENVScreen();
  } else {
    drawGasScreen();
  }

  delay(1000); // Wait 1 second before next reading and update
}