// ACS712 AC Current + Frequency Monitor (Manual Calculation for Accuracy)
// Uses your exact calibration and RMS method
// Includes Rob Tillaart ACS712 library ONLY for frequency detection
// Sends: current_mA (int rounded), current_A (3 decimals), frequency_Hz (1 decimal)
// Example: 1234,1.234,50.1

#include <ACS712.h>

const int SENSOR_PIN = 4;
const float SENSOR_VCC = 5.0;
const float ADC_VCC = 3.3;
const float SENSITIVITY = 0.185;              // V/A for 5A version
const float DIVIDER_RATIO = 10.0 / (22.0 + 10.0);  // ≈0.3125
const float NOISE_THRESHOLD_MA = 150.0;

// Library object ONLY for frequency detection (we use manual current calc)
ACS712 freqSensor(SENSOR_PIN, 5.0, 4095, 185);  // Parameters don't affect freq detection

float OFFSET = SENSOR_VCC / 2.0;  // ~2.5V initial

void setup() {
  Serial.begin(115200);
  Serial.println("ACS712 AC Monitor Starting...");
  Serial.println("Calibrating offset (NO CURRENT!)...");

  OFFSET = calibrateOffset();

  Serial.print("Calibrated Offset: ");
  Serial.print(OFFSET, 4);
  Serial.println(" V");

  // Optional: library auto calibration for freq (helps zero-crossing)
  freqSensor.autoMidPoint();

  Serial.println("Monitoring AC RMS Current + Frequency");
  Serial.println("Format: mA,A,Hz");
}

float calibrateOffset() {
  long sum = 0;
  int samples = 1000;
  for (int i = 0; i < samples; i++) {
    sum += analogRead(SENSOR_PIN);
    delayMicroseconds(100);
  }
  float rawAvg = sum / (float)samples;
  float measuredV = (rawAvg / 4095.0) * ADC_VCC;
  return measuredV / DIVIDER_RATIO;
}

float measureRMSCurrent() {
  float sumSquares = 0.0;
  int samples = 300;  // Slightly more than your 200 for better accuracy
  for (int i = 0; i < samples; i++) {
    int rawValue = analogRead(SENSOR_PIN);
    float measuredV = (rawValue / 4095.0) * ADC_VCC;
    float sensorV = measuredV / DIVIDER_RATIO;
    float instantCurrent = (sensorV - OFFSET) / SENSITIVITY;
    sumSquares += instantCurrent * instantCurrent;
    delayMicroseconds(200);
  }
  return sqrt(sumSquares / samples);  // RMS current in A
}

void loop() {
  float currentA = measureRMSCurrent();

  // Noise filtering (exactly like your code)
  float current_mA = currentA * 1000.0;
  if (current_mA < NOISE_THRESHOLD_MA) {
    current_mA = 0.0;
    currentA = 0.0;
  }

  // Frequency detection using library (robust zero-crossing)
  float frequency = freqSensor.detectFrequency(45);  // Search from 45Hz

  // Send data
  Serial.print((int)round(current_mA));
  Serial.print(",");
  Serial.print(currentA, 3);
  Serial.print(",");
  Serial.println(frequency, 1);

  delay(1000);  // 1 second update
}