
#include <WiFi.h>
#include <Firebase_ESP_Client.h>
#include <Wire.h>
#include <Adafruit_INA219.h>

// WiFi credentials
#define WIFI_SSID "vivo"
#define WIFI_PASSWORD "muru555999"

// Firebase credentials
#define API_KEY "AIzaSyCtkhhd_AuunnmePvratW6Wk_vSVfAbtPU"
#define DATABASE_URL "https://esp32camsecurity-default-rtdb.asia-southeast1.firebasedatabase.app/"

// Sensor pins
#define ZMPT_PIN 32
#define ACS712_GEN1_PIN 34
#define ACS712_GEN2_PIN 35
#define RELAY_FAULT_PIN 26

// Calibration constants
#define ZMPT_CALIBRATION 15.26
//#define ZMPT_OFFSET 2048
#define ZMPT_OFFSET 0
#define ACS712_SENSITIVITY 0.185
#define ACS712_OFFSET 2500

// Firebase objects
FirebaseData fbdo;
FirebaseAuth auth;
FirebaseConfig config;

Adafruit_INA219 ina219;

unsigned long lastSendTime = 0;
bool signupOK = false;

float readACS712Current(int pin) {
  int raw = analogRead(pin);
  float voltage = (raw / 4095.0) * 3300.0;
  float current = (voltage - ACS712_OFFSET) / (ACS712_SENSITIVITY * 1000.0);
  return current;
}

float readZMPTVoltage() {
  int raw = analogRead(ZMPT_PIN);
  float voltage = (raw - ZMPT_OFFSET) * (3.3 / 4095.0) * ZMPT_CALIBRATION;
  return abs(voltage);
}

float readFrequency() {
  int threshold = 2048;
  unsigned long startTime = millis();
  int crossings = 0;
  bool prevHigh = false;

  while (millis() - startTime < 1000) {
    int value = analogRead(ZMPT_PIN);
    bool high = value > threshold;
    if (high && !prevHigh) {
      crossings++;
    }
    prevHigh = high;
    delayMicroseconds(500);
  }
  return crossings / 2.0;
}

void setup() {
  Serial.begin(115200);
  pinMode(RELAY_FAULT_PIN, OUTPUT);
  digitalWrite(RELAY_FAULT_PIN, HIGH);

  WiFi.begin(WIFI_SSID, WIFI_PASSWORD);
  Serial.print("Connecting to WiFi");
  while (WiFi.status() != WL_CONNECTED) {
    delay(300);
    Serial.print(".");
  }
  Serial.println("\nWiFi connected");

  config.api_key = API_KEY;
  config.database_url = DATABASE_URL;
  // NTP time sync required for SSL
configTime(0, 0, "pool.ntp.org", "time.nist.gov");
Serial.print("Waiting for time sync");
time_t now = time(nullptr);
while (now < 8 * 3600 * 2) {
  delay(500);
  Serial.print(".");
  now = time(nullptr);
}
Serial.println("\nTime synchronized");

  if (Firebase.signUp(&config, &auth, "", "")) {
    Serial.println("Firebase signup successful");
    signupOK = true;
  } else {
    Serial.printf("Firebase signup failed: %s\n", config.signer.signupError.message.c_str());
  }

  Firebase.begin(&config, &auth);
  Firebase.reconnectWiFi(true);

  if (!ina219.begin()) {
    Serial.println("INA219 not detected!");
    while (1);
  }
}

void loop() {
  if (Firebase.ready() && signupOK && millis() - lastSendTime > 2000) {
    lastSendTime = millis();

    float genVoltage = readZMPTVoltage();
    float gen1Current = readACS712Current(ACS712_GEN1_PIN);
    float gen2Current = readACS712Current(ACS712_GEN2_PIN);
    float loadVoltage = ina219.getBusVoltage_V();
    float loadCurrent = ina219.getCurrent_mA() / 1000.0;
    float frequency = readFrequency();

    float genPower = genVoltage * (gen1Current + gen2Current);
    float loadPower = loadVoltage * loadCurrent;

    bool fault = false;
    String faultReason = "None";

    if (frequency > 51.0) {
      fault = true;
      faultReason = "Over Frequency Fault";
    } else if (frequency < 45.0) {
      fault = true;
      faultReason = "Under Frequency Fault";
    } else if (loadVoltage > 23.0) {
      fault = true;
      faultReason = "Over Voltage Fault";
    } else if (loadVoltage < 2.0 && loadCurrent > 0.5) {
      fault = true;
      faultReason = "Short Circuit Fault";
    } else if (loadPower > 9.0) {
      fault = true;
      faultReason = "Overload Fault";
    }
// Reset all alarms
Firebase.RTDB.setBool(&fbdo, "/SmartGrid/Fault/Alarm/Overload", false);
Firebase.RTDB.setBool(&fbdo, "/SmartGrid/Fault/Alarm/OverFrequency", false);
Firebase.RTDB.setBool(&fbdo, "/SmartGrid/Fault/Alarm/UnderFrequency", false);
Firebase.RTDB.setBool(&fbdo, "/SmartGrid/Fault/Alarm/ShortCircuit", false);
Firebase.RTDB.setBool(&fbdo, "/SmartGrid/Fault/Alarm/OverVoltage", false);
Firebase.RTDB.setBool(&fbdo, "/SmartGrid/Fault/Alarm/CableEarthFault", false);
Firebase.RTDB.setBool(&fbdo, "/SmartGrid/Fault/Alarm/LineToLineFault", false);

// Set specific alarm based on reason
if (faultReason == "Overload Fault") {
  Firebase.RTDB.setBool(&fbdo, "/SmartGrid/Fault/Alarm/Overload", true);
} else if (faultReason == "Over Frequency Fault") {
  Firebase.RTDB.setBool(&fbdo, "/SmartGrid/Fault/Alarm/OverFrequency", true);
} else if (faultReason == "Under Frequency Fault") {
  Firebase.RTDB.setBool(&fbdo, "/SmartGrid/Fault/Alarm/UnderFrequency", true);
} else if (faultReason == "Short Circuit Fault") {
  Firebase.RTDB.setBool(&fbdo, "/SmartGrid/Fault/Alarm/ShortCircuit", true);
} else if (faultReason == "Over Voltage Fault") {
  Firebase.RTDB.setBool(&fbdo, "/SmartGrid/Fault/Alarm/OverVoltage", true);
}

    // Upload core data
    Firebase.RTDB.setFloat(&fbdo, "/SmartGrid/Generation/Voltage", genVoltage);
    Firebase.RTDB.setFloat(&fbdo, "/SmartGrid/Generation/Current_Gen1", gen1Current);
    Firebase.RTDB.setFloat(&fbdo, "/SmartGrid/Generation/Current_Gen2", gen2Current);
    Firebase.RTDB.setFloat(&fbdo, "/SmartGrid/Generation/Power", genPower);

    Firebase.RTDB.setFloat(&fbdo, "/SmartGrid/Load/Voltage", loadVoltage);
    Firebase.RTDB.setFloat(&fbdo, "/SmartGrid/Load/Current", loadCurrent);
    Firebase.RTDB.setFloat(&fbdo, "/SmartGrid/Load/Power", loadPower);
    Firebase.RTDB.setFloat(&fbdo, "/SmartGrid/Frequency", frequency);
    Firebase.RTDB.setString(&fbdo, "/SmartGrid/Fault/Status", fault ? "Fault Detected" : "No Fault");
    Firebase.RTDB.setString(&fbdo, "/SmartGrid/Fault/Reason", faultReason);

    // Auto/manual relay mode
    bool autoMode = true;
    if (Firebase.RTDB.getBool(&fbdo, "/SmartGrid/Remote_Control/Auto_Mode")) {
      autoMode = fbdo.boolData();
    }

    if (autoMode) {
      digitalWrite(RELAY_FAULT_PIN, fault ? LOW : HIGH);
    } else {
      if (Firebase.RTDB.getInt(&fbdo, "/SmartGrid/Remote_Control/Fault_Relay")) {
        int relayState = fbdo.intData();
        digitalWrite(RELAY_FAULT_PIN, relayState == 1 ? LOW : HIGH);
      }
    }

    // Optional Load Shedding
 if (autoMode) {
  if (fault) {
    Firebase.RTDB.setInt(&fbdo, "/SmartGrid/Remote_Control/Unimportant_Load_Relay", 1);
    // no delay here — conditionally trip important load next cycle
  } else {
    Firebase.RTDB.setInt(&fbdo, "/SmartGrid/Remote_Control/Unimportant_Load_Relay", 0);
    Firebase.RTDB.setInt(&fbdo, "/SmartGrid/Remote_Control/Important_Load_Relay", 0);
  }
}


    // --- FAST RELAY READ + LOAD POWER CALCULATION ---
    int importantRelay = Firebase.RTDB.getInt(&fbdo, "/SmartGrid/Remote_Control/Important_Load_Relay") ? fbdo.intData() : 0;
    int unimportantRelay = Firebase.RTDB.getInt(&fbdo, "/SmartGrid/Remote_Control/Unimportant_Load_Relay") ? fbdo.intData() : 0;

    float powerImportant = 0.0;
    float powerUnimportant = 0.0;

    if (importantRelay == 1 && unimportantRelay == 1) {
      powerImportant = loadPower / 2.0;
      powerUnimportant = loadPower / 2.0;
    } else if (importantRelay == 1) {
      powerImportant = loadPower;
      powerUnimportant = 0.0;
    } else if (unimportantRelay == 1) {
      powerImportant = 0.0;
      powerUnimportant = loadPower;
    }

    Firebase.RTDB.setFloat(&fbdo, "/SmartGrid/Load/Power_Important_Load", powerImportant);
    Firebase.RTDB.setFloat(&fbdo, "/SmartGrid/Load/Power_Unimportant_Load", powerUnimportant);

    // --- Debug Log ---
    Serial.println("---- Data Log ----");
    Serial.print("Gen Voltage: "); Serial.println(genVoltage);
    Serial.print("Gen1 Current: "); Serial.println(gen1Current);
    Serial.print("Gen2 Current: "); Serial.println(gen2Current);
    Serial.print("Load Voltage: "); Serial.println(loadVoltage);
    Serial.print("Load Current: "); Serial.println(loadCurrent);
    Serial.print("Frequency: "); Serial.println(frequency);
    Serial.print("Load Power: "); Serial.println(loadPower);
    Serial.print("Important Power: "); Serial.println(powerImportant);
    Serial.print("Unimportant Power: "); Serial.println(powerUnimportant);
    Serial.print("Fault: "); Serial.println(faultReason);
    Serial.print("Relay Output (GPIO 26): "); Serial.println(digitalRead(RELAY_FAULT_PIN));
    Serial.println("------------------\n");
  }
}


