#include <WiFi.h>
#include <WebServer.h>
#include <EEPROM.h>

// WiFi credentials for STA mode
const char* sta_ssid = "JioFiber-4G";
const char* sta_password = "1234567890";

// AP mode credentials
const char* ap_ssid = "WaterMeter";
const char* ap_password = "meter123";

// Pin definitions for ESP32-S2
const int turbidityPin = 1;   // Turbidity sensor analog pin (ADC) - but used for level? Adjust if needed
const int levelPins[] = {3, 4, 5};  // Digital pins for level sensing via conductivity (3 levels)
const int numLevels = sizeof(levelPins) / sizeof(int);
const int tdsPin = 6;         // Analog pin for TDS via conductivity (voltage divider)
const int pumpPin = 33;       // Pump control

// Thresholds
const int tdsThreshold = 500;   // ppm
const int turbThreshold = 100;  // NTU

// TDS calculation constants (calibrate these)
const float seriesResistor = 10000.0;  // 10kΩ series resistor
const float cellConstant = 1.0;        // cm^-1, calibrate
const float tdsFactor = 0.7;           // For TDS = EC * factor

// Global variables
float tankSize = 1000.0;       // Tank size in liters
float tankLevelPercent = 0.0;  // From sensors
int tdsValue = 0;              // TDS in ppm
float tempC = 25.0;            // Temperature in C (simulated)
float turbidityNTU = 0.0;      // Turbidity
bool pumpOn = false;
bool highAlert = false;
String homeName = "Home1";
String alertMsg = "";          // For level-based alerts

// Simulated DB - In-memory array for simplicity (up to 10 entries)
struct DataEntry {
  String timestamp;
  String home;
  float level;
  float litres;
  int tds;
  float turbidity;
  float temp;
};
DataEntry dbEntries[10];
int dbCount = 0;

WebServer server(80);

unsigned long lastPrintTime = 0;
const unsigned long printInterval = 5000; // Print to Serial every 5 seconds
unsigned long lastSensorTime = 0;
const unsigned long sensorInterval = 1000; // Update sensors every second

float getLevel() {
  int count = 0;
  for (int i = 0; i < numLevels; i++) {
    if (digitalRead(levelPins[i]) == LOW) {  // Detect conductivity (water connects to GND)
      count++;
    }
  }
  return (static_cast<float>(count) / numLevels) * 100.0;
}

void updateSensors() {
  // Read digital pins for water level (LOW = wet via conductivity to GND)
  tankLevelPercent = getLevel();
  
  // TDS via conductivity on analog pin 6 (voltage divider)
  int rawTDS = analogRead(tdsPin);
  float voltage = (rawTDS / 4095.0) * 3.3;
  if (voltage > 0 && voltage < 3.3) {
    float rWater = seriesResistor * (3.3 / voltage - 1);
    float ec = 1000000.0 / (rWater * cellConstant);  // µS/cm
    tdsValue = (int)(ec * tdsFactor);  // ppm
    tdsValue = constrain(tdsValue, 50, 1000);  // Clamp
  } else {
    tdsValue = 0;  // Error or dry
  }
  
  // Turbidity from ADC (high voltage = low turbidity, calibrate) - on pin 1, but if pin 1 is level, move turbidity to another analog
  int rawTurb = analogRead(turbidityPin);
  turbidityNTU = constrain(map(rawTurb, 0, 4095, 1000, 0), 0, 1000);
  
  // Temp simulated for now
  tempC = random(150, 350) / 10.0;  // 15-35 C
  
  // Level-based alert
  if (tankLevelPercent <= 0) {
    alertMsg = "Dry Alert! No water.";
  } else if (tankLevelPercent <= 15) {
    alertMsg = "Attention: Reserved water low (15%).";
  } else if (tankLevelPercent <= 50) {
    alertMsg = "50% Alert: Half full.";
  } else if (tankLevelPercent >= 100) {
    alertMsg = "Tank Full (100%).";
  } else {
    alertMsg = "";
  }
  
  // Overall high alert for TDS/Turb
  highAlert = (tdsValue > tdsThreshold) || (turbidityNTU > turbThreshold) || !alertMsg.isEmpty();
  
  // Debug print raw values
  Serial.print("Raw TDS ADC: "); Serial.print(rawTDS);
  Serial.print(" | Voltage: "); Serial.print(voltage);
  Serial.print(" | TDS: "); Serial.println(tdsValue);
  for (int i = 0; i < numLevels; i++) {
    Serial.print("Level Pin "); Serial.print(levelPins[i]); 
    Serial.print(": "); Serial.println(digitalRead(levelPins[i]));
  }
}

void logToDB() {
  float litres = (tankLevelPercent / 100.0) * tankSize;
  if (dbCount < 10) {
    dbEntries[dbCount].timestamp = String(millis() / 1000) + "s";
    dbEntries[dbCount].home = homeName;
    dbEntries[dbCount].level = tankLevelPercent;
    dbEntries[dbCount].litres = litres;
    dbEntries[dbCount].tds = tdsValue;
    dbEntries[dbCount].turbidity = turbidityNTU;
    dbEntries[dbCount].temp = tempC;
    dbCount++;
  } else {
    // Shift array (overwrite oldest)
    for (int i = 0; i < 9; i++) {
      dbEntries[i] = dbEntries[i + 1];
    }
    dbEntries[9].timestamp = String(millis() / 1000) + "s";
    dbEntries[9].home = homeName;
    dbEntries[9].level = tankLevelPercent;
    dbEntries[9].litres = litres;
    dbEntries[9].tds = tdsValue;
    dbEntries[9].turbidity = turbidityNTU;
    dbEntries[9].temp = tempC;
  }
}

const char htmlPage[] PROGMEM = R"rawliteral(
<!DOCTYPE html>
<html>
<head>
  <title>Advanced Water Meter</title>
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <style>
    body { background: white; font-family: Arial, sans-serif; margin: 0; padding: 20px; }
    .container { display: flex; flex-wrap: wrap; gap: 20px; justify-content: center; }
    .tank { position: relative; width: 200px; height: 300px; border: 3px solid black; background: white; }
    .water { position: absolute; bottom: 0; left: 0; width: 100%; transition: height 0.5s, background-color 0.5s; }
    .waves { position: absolute; left: 0; width: 100%; height: 20px; background: radial-gradient(circle, transparent 20%, lightblue 20%); background-size: 20px 20px; animation: wave 2s linear infinite; }
    @keyframes wave { 0% { background-position: 0 0; } 100% { background-position: 20px 0; } }
    .overlay { position: absolute; top: 10px; left: 50%; transform: translateX(-50%); background: rgba(255, 255, 255, 0.5); padding: 5px 10px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0, 0, 0, 0.2); font-size: 14px; color: #333; }
    .card { background: white; padding: 20px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); width: 200px; text-align: center; }
    .value { font-size: 20px; font-weight: bold; }
    .cards-row { display: flex; gap: 20px; flex-wrap: wrap; justify-content: center; }
    .controls { display: flex; flex-direction: column; gap: 10px; align-items: center; margin-top: 20px; }
    button { padding: 10px 20px; background: #007bff; color: white; border: none; border-radius: 5px; cursor: pointer; }
    button:hover { background: #0056b3; }
    input { padding: 10px; border: 1px solid #ccc; border-radius: 5px; }
    #dbTable { display: none; width: 100%; border-collapse: collapse; margin-top: 20px; }
    #dbTable th, #dbTable td { border: 1px solid #ddd; padding: 8px; text-align: left; }
    #dbTable th { background: #f2f2f2; }
    .pump-status { position: fixed; top: 10px; right: 10px; width: 50px; height: 50px; border-radius: 50%; display: flex; justify-content: center; align-items: center; color: white; font-size: 12px; box-shadow: 0 2px 10px rgba(0,0,0,0.2); transition: background 0.3s; }
    #alertCard { background: red; color: white; padding: 15px; border-radius: 10px; margin: 10px; display: none; text-align: center; font-weight: bold; }
  </style>
</head>
<body>
  <div class="container">
    <div class="tank">
      <div class="water" id="water"></div>
      <div class="waves" id="waves"></div>
      <div class="overlay" id="level">0%</div>
    </div>
    <div id="alertCard">🚨 Alert: <span id="alertMsg"></span></div>
    <div class="cards-row">
      <div class="card">
        <h3>TDS</h3>
        <div><span class="value" id="tds">0 ppm</span></div>
      </div>
      <div class="card">
        <h3>Turbidity</h3>
        <div><span class="value" id="turbidity">0 NTU</span></div>
      </div>
      <div class="card">
        <h3>Temperature</h3>
        <div><span class="value" id="temp">0 C</span></div>
      </div>
      <div class="card">
        <h3>Water Level</h3>
        <div><span class="value" id="litres">0 L</span></div>
        <div><span id="levelPercent">0%</span></div>
      </div>
    </div>
    <div class="controls">
      <button onclick="togglePump()">Toggle Pump</button>
      <div>
        Tank Size (litres): <input id="size" type="number" value="1000">
        <button onclick="setSize()">Set</button>
      </div>
      <button onclick="showDB()">Check DB</button>
    </div>
  </div>
  <table id="dbTable">
    <thead><tr><th>Time</th><th>Home</th><th>Level %</th><th>Litres</th><th>TDS ppm</th><th>Turbidity NTU</th><th>Temp C</th></tr></thead>
    <tbody id="dbBody"></tbody>
  </table>
  <div class="pump-status" id="pumpStatus">OFF</div>
  <script>
    function updateData() {
      fetch('/data')
        .then(r => r.json())
        .then(d => {
          const percent = d.level.toFixed(1);
          const litres = d.litres.toFixed(1);
          document.getElementById('level').innerText = percent + '%';
          document.getElementById('levelPercent').innerText = percent + '%';
          document.getElementById('litres').innerText = litres + ' L';
          document.getElementById('tds').innerText = d.tds;
          document.getElementById('turbidity').innerText = d.turbidity.toFixed(0);
          document.getElementById('temp').innerText = d.temp.toFixed(1);
          document.getElementById('water').style.height = percent + '%';
          const waveTop = 300 - (300 * (percent / 100)) - 20;
          document.getElementById('waves').style.top = waveTop + 'px';
          
          // Water color based on turbidity
          let color;
          if (d.turbidity < 50) color = 'lightblue';
          else if (d.turbidity < 200) color = '#b0c4de';
          else color = '#8b4513';
          document.getElementById('water').style.backgroundColor = color;
          
          // Alert
          document.getElementById('alertMsg').innerText = d.alertMsg;
          document.getElementById('alertCard').style.display = d.highAlert ? 'block' : 'none';
          
          // Update pump status
          const status = document.getElementById('pumpStatus');
          status.innerText = d.pumpOn ? 'ON' : 'OFF';
          status.style.background = d.pumpOn ? 'green' : 'red';
        });
    }

    function togglePump() {
      fetch('/togglePump').then(() => updateData());
    }

    function setSize() {
      const newSize = document.getElementById('size').value;
      fetch('/setSize?size=' + newSize).then(() => updateData());
    }

    function showDB() {
      fetch('/db')
        .then(r => r.json())
        .then(data => {
          const tbody = document.getElementById('dbBody');
          tbody.innerHTML = '';
          data.forEach(entry => {
            const row = `<tr><td>${entry.timestamp}</td><td>${entry.home}</td><td>${entry.level.toFixed(1)}</td><td>${entry.litres.toFixed(1)}</td><td>${entry.tds}</td><td>${entry.turbidity.toFixed(0)}</td><td>${entry.temp.toFixed(1)}</td></tr>`;
            tbody.innerHTML += row;
          });
          document.getElementById('dbTable').style.display = 'table';
        });
    }

    setInterval(updateData, 1000);
    updateData();
  </script>
</body>
</html>
)rawliteral";

void handleRoot() {
  server.send_P(200, "text/html", htmlPage);
}

void handleData() {
  updateSensors();
  float litres = (tankLevelPercent / 100.0) * tankSize;
  String json = "{\"level\":" + String(tankLevelPercent) + 
                ",\"litres\":" + String(litres) +
                ",\"tds\":" + String(tdsValue) +
                ",\"turbidity\":" + String(turbidityNTU) +
                ",\"temp\":" + String(tempC) +
                ",\"pumpOn\":" + (pumpOn ? "true" : "false") +
                ",\"highAlert\":" + (highAlert ? "true" : "false") +
                ",\"alertMsg\":\"" + alertMsg + "\"}";
  server.send(200, "application/json", json);
}

void handleTogglePump() {
  pumpOn = !pumpOn;
  digitalWrite(pumpPin, pumpOn ? HIGH : LOW);
  server.send(200, "text/plain", pumpOn ? "ON" : "OFF");
}

void handleSetSize() {
  if (server.hasArg("size")) {
    tankSize = server.arg("size").toFloat();
  }
  server.send(200, "text/plain", "OK");
}

void handleDB() {
  String json = "[";
  for (int i = 0; i < dbCount; i++) {
    if (i > 0) json += ",";
    json += "{\"timestamp\":\"" + dbEntries[i].timestamp + 
            "\",\"home\":\"" + dbEntries[i].home + 
            "\",\"level\":" + String(dbEntries[i].level) + 
            ",\"litres\":" + String(dbEntries[i].litres) + 
            ",\"tds\":" + String(dbEntries[i].tds) + 
            ",\"turbidity\":" + String(dbEntries[i].turbidity) + 
            ",\"temp\":" + String(dbEntries[i].temp) + "}";
  }
  json += "]";
  server.send(200, "application/json", json);
}

void printStatus() {
  float litres = (tankLevelPercent / 100.0) * tankSize;
  Serial.println("--- Water Meter Status ---");
  Serial.print("Level: "); Serial.print(tankLevelPercent); Serial.print("% ("); Serial.print(litres); Serial.println(" L)");
  Serial.print("TDS: "); Serial.println(tdsValue);
  Serial.print("Turbidity: "); Serial.println(turbidityNTU); Serial.println(" NTU");
  Serial.print("Temp: "); Serial.println(tempC); Serial.println(" C");
  Serial.print("Alert: "); Serial.println(highAlert ? "HIGH: " + alertMsg : "OK");
  Serial.print("Pump: "); Serial.println(pumpOn ? "ON" : "OFF");
  Serial.print("DB Entries: "); Serial.println(dbCount);
  Serial.println("-------------------------");
}

void setup() {
  Serial.begin(115200);
  EEPROM.begin(512);

  pinMode(pumpPin, OUTPUT);
  digitalWrite(pumpPin, LOW);

  // Set up level pins for conductivity sensing
  for (int i = 0; i < numLevels; i++) {
    pinMode(levelPins[i], INPUT_PULLUP);
  }

  // WiFi AP + STA mode
  WiFi.mode(WIFI_AP_STA);
  WiFi.softAP(ap_ssid, ap_password);
  Serial.print("AP IP: "); Serial.println(WiFi.softAPIP());
  
  WiFi.begin(sta_ssid, sta_password);
  Serial.print("Connecting to STA ");
  int attempts = 0;
  while (WiFi.status() != WL_CONNECTED && attempts < 20) {
    delay(500);
    Serial.print(".");
    attempts++;
  }
  if (WiFi.status() == WL_CONNECTED) {
    Serial.println();
    Serial.print("STA Connected! IP: "); Serial.println(WiFi.localIP());
  } else {
    Serial.println();
    Serial.println("STA Failed to connect.");
  }

  server.on("/", handleRoot);
  server.on("/data", handleData);
  server.on("/togglePump", handleTogglePump);
  server.on("/setSize", handleSetSize);
  server.on("/db", handleDB);

  server.begin();
  lastSensorTime = millis();
}

void loop() {
  server.handleClient();

  unsigned long now = millis();
  if (now - lastSensorTime >= sensorInterval) {
    updateSensors();
    logToDB();
    lastSensorTime = now;
  }

  if (now - lastPrintTime >= printInterval) {
    printStatus();
    lastPrintTime = now;
  }

  delay(10);
}