#include <Wire.h>
#include <VL53L0X.h>
#include <Adafruit_SSD1306.h>
#include <OneWire.h>
#include <DallasTemperature.h>
// ---------------- OLED ----------------
#define SCREEN_WIDTH 128
#define SCREEN_HEIGHT 32
Adafruit_SSD1306 display(SCREEN_WIDTH, SCREEN_HEIGHT, &Wire, -1);
// ---------------- VL53L0X ----------------
VL53L0X sensor;
bool vl53Present = false;
// ------- VL53 smoothing -------
const uint8_t VL53_MEDIAN_N = 5;
const int VL53_MAX_STEP = 20;
const int VL53_JUMP_REJECT = 150;
const int VL53_MIN = 30;
const int VL53_MAX = 2000;
int lastValid = -1;
float emaDist = -1;
const float EMA_ALPHA = 0.25;
// helper: read raw once
int readVL53RawOnce() {
int d = sensor.readRangeContinuousMillimeters();
if (sensor.timeoutOccurred()) return -1;
if (d < VL53_MIN || d > VL53_MAX) return -1;
return d;
}
// helper: median of N samples
int medianOfSamples() {
int buf[VL53_MEDIAN_N];
uint8_t got = 0;
for (uint8_t i=0; i<VL53_MEDIAN_N; i++) {
int v = readVL53RawOnce();
if (v >= 0) buf[got++] = v;
}
if (got == 0) return -1;
for (uint8_t i=1; i<got; i++) {
int key = buf[i]; int j = i;
while (j>0 && buf[j-1] > key) { buf[j] = buf[j-1]; j--; }
buf[j] = key;
}
return buf[got/2];
}
// main smoother
int readVL53Smooth() {
if (!vl53Present) return -1;
int med = medianOfSamples();
if (med < 0) return (emaDist < 0 ? -1 : (int)round(emaDist));
if (lastValid >= 0 && abs(med - lastValid) > VL53_JUMP_REJECT) {
return (emaDist < 0 ? lastValid : (int)round(emaDist));
}
lastValid = med;

if (emaDist < 0) emaDist = med;
else emaDist = emaDist + EMA_ALPHA * (med - emaDist);
static int reported = -1;
int candidate = (int)round(emaDist);
if (reported < 0) reported = candidate;
int delta = candidate - reported;
if (delta > VL53_MAX_STEP) reported += VL53_MAX_STEP;
else if (delta < -VL53_MAX_STEP) reported -= VL53_MAX_STEP;
else reported = candidate;
static int lastOut = -10000;
if (abs(reported - lastOut) == 1) reported = lastOut;
lastOut = reported;
return reported;
}
// ---------------- MPU6050 ----------------
#define MPU_ADDR 0x68
#define TRACK_GAUGE 1676.0
float Q_angle = 0.00005;
float Q_bias = 0.0001;
float R_measure = 0.005;
float angle = 0;
float bias = 0;
float P[2][2] = {{0,0},{0,0}};
unsigned long lastTime;
float kalmanUpdate(float newAngle, float newRate, float dt) {
angle += dt * (newRate - bias);
P[0][0] += dt * (dt*P[1][1] - P[0][1] - P[1][0] + Q_angle);
P[0][1] -= dt * P[1][1];
P[1][0] -= dt * P[1][1];
P[1][1] += Q_bias * dt;
float S = P[0][0] + R_measure;
float K[2];
K[0] = P[0][0] / S;
K[1] = P[1][0] / S;
float y = newAngle - angle;
angle += K[0] * y;
bias += K[1] * y;
float P00_temp = P[0][0];
float P01_temp = P[0][1];
P[0][0] -= K[0] * P00_temp;
P[0][1] -= K[0] * P01_temp;
P[1][0] -= K[1] * P00_temp;
P[1][1] -= K[1] * P01_temp;
return angle;
}
void writeReg(uint8_t reg, uint8_t data) {
Wire.beginTransmission(MPU_ADDR);
Wire.write(reg);
Wire.write(data);
Wire.endTransmission();
}
void readRaw(int16_t& ax, int16_t& ay, int16_t& az, int16_t& gx, int16_t& gy, int16_t& gz) {
Wire.beginTransmission(MPU_ADDR);
Wire.write(0x3B);
Wire.endTransmission(false);

Wire.requestFrom(MPU_ADDR, 14, true);
ax = Wire.read() << 8 | Wire.read();
ay = Wire.read() << 8 | Wire.read();
az = Wire.read() << 8 | Wire.read();
Wire.read(); Wire.read();
gx = Wire.read() << 8 | Wire.read();
gy = Wire.read() << 8 | Wire.read();
gz = Wire.read() << 8 | Wire.read();
}
// ---- Filters ----
#define MEDIAN_SIZE 7
#define AVG_SIZE 10
float seBuffer[MEDIAN_SIZE];
float avgBuffer[AVG_SIZE];
byte seIndex = 0, avgIndex = 0;
float medianFilter(float val) {
seBuffer[seIndex] = val;
seIndex = (seIndex + 1) % MEDIAN_SIZE;
float sorted[MEDIAN_SIZE];
for (byte i=0; i<MEDIAN_SIZE; i++) sorted[i] = seBuffer[i];
for (byte i=1; i<MEDIAN_SIZE; i++) {
float key = sorted[i]; int j = i;
while (j>0 && sorted[j-1] > key) { sorted[j] = sorted[j-1]; j--; }
sorted[j] = key;
}
return sorted[MEDIAN_SIZE/2];
}
float movingAverage(float val) {
avgBuffer[avgIndex] = val;
avgIndex = (avgIndex + 1) % AVG_SIZE;
float sum = 0;
for (byte i=0; i<AVG_SIZE; i++) sum += avgBuffer[i];
return sum / AVG_SIZE;
}
// ---------------- DS18B20 ----------------
#define ONE_WIRE_BUS 3 // DS18B20 data pin
OneWire oneWire(ONE_WIRE_BUS);
DallasTemperature sensors(&oneWire);
// ---------------- Button ----------------
#define BTN_PIN 8
byte mode = 0; // 0=SE, 1=Gauge, 2=Temp
// ---------------- SETUP ----------------
void setup() {
Serial.begin(115200);
Wire.begin();
pinMode(BTN_PIN, INPUT_PULLUP);
sensors.begin();
// OLED init
if (!display.begin(SSD1306_SWITCHCAPVCC, 0x3C)) {
Serial.println("SSD1306 failed");
for(;;);
}
display.clearDisplay();
display.setTextColor(SSD1306_WHITE);
// MPU6050 wake

writeReg(0x6B, 0x00);
lastTime = micros();
// VL53L0X init
sensor.setTimeout(500);
if (sensor.init()) {
vl53Present = true;
sensor.setMeasurementTimingBudget(66000);
sensor.startContinuous();
Serial.println("VL53L0X detected");
} else {
vl53Present = false;
Serial.println("VL53L0X not detected -> Running in SE-only mode");
}
// ---- Show startup message ----
display.clearDisplay();
display.setTextSize(2);
display.setCursor(0, 0);
display.println("Rail Sathi");
display.setCursor(0, 16);
display.println(" V 1.0");
display.display();
delay(3000);
display.clearDisplay();
display.display();
}
// ---------------- LOOP ----------------
void loop() {
// ---- Button Handling ----
static bool lastBtnState = HIGH;
bool btnState = digitalRead(BTN_PIN);
if (lastBtnState == HIGH && btnState == LOW) {
mode = (mode + 1) % 3; // cycle modes
delay(200);
}
lastBtnState = btnState;
// ---- MPU Roll + Superelevation ----
int16_t ax, ay, az, gx, gy, gz;
readRaw(ax, ay, az, gx, gy, gz);
float accelRoll = atan2((float)ay, (float)az) * 180.0 / PI;
float gyroRate = gx / 131.0;
unsigned long now = micros();
float dt = (now - lastTime) / 1000000.0;
lastTime = now;
float roll = kalmanUpdate(accelRoll, gyroRate, dt);
float rollPrec = round(roll * 10.0) / 100.0;
float rollRad = rollPrec * PI / 180.0;
float superelevation = TRACK_GAUGE * tan(rollRad);
float seMedian = medianFilter(superelevation);
float seSmoothed = movingAverage(seMedian);
int seInt = abs((int)round(seSmoothed));
if (seInt < 1) seInt = 0;
char side = (seInt > 0) ? ((roll >= 0) ? 'L' : 'R') : ' ';
// ---- VL53L0X Distance ----
int distance = vl53Present ? readVL53Smooth() : -1;
// ---- OLED Display ----

display.clearDisplay();
display.setTextSize(2);
if (mode == 0) {
// ---- SE Mode ----
display.setCursor(0,0);
display.print("SE: ");
display.print(seInt);
display.print(" ");
if (seInt > 0) display.print(side);
} else if (mode == 1) {
// ---- Gauge Mode ----
display.setCursor(0,0);
display.print("Gauge:");
display.setCursor(0,16);
if (distance >= 0) display.print(42 - distance);
else display.print("N.C.");
display.print(" mm");
} else if (mode == 2) {
// ---- Temperature Mode ----
sensors.requestTemperatures();
float tempC = sensors.getTempCByIndex(0);
display.setCursor(0,0);
display.print("Temp:");
display.setCursor(0,16);
if (tempC != DEVICE_DISCONNECTED_C) {
display.print(tempC,1);
display.print((char)247);
display.print("C");
} else {
display.print("N.C."); // sensor is not connected.
}
}
display.display();
delay(100);
}