// ESP RainMaker + FastLED – Full RGB Lights + Fan + Switch + Touch + GPIO0 Reset (EXACTLY like original example)
#include "RMaker.h"
#include "WiFi.h"
#include "WiFiProv.h"
#include "AppInsights.h"
#include <FastLED.h>

#define TOUCH_THRESHOLD 100

const char *service_name = "PROV_RGB_HOME";
const char *pop = "abcd1234";

// MUST be constexpr or #define for FastLED template
constexpr uint8_t LIGHT1_PIN = 11;   // WS2812B Data – Light 1
constexpr uint8_t LIGHT2_PIN = 33; // WS2812B Data – Light 2

static int gpio_reset   = 0;   // GPIO0 – Boot button (handled exactly like official example)
static int fan_pin      = 10;
static int switch_pin   = 41;

// Touch pins
static int touch_fan    = 1;  // T1
static int touch_light1 = 3;  // T2
static int touch_light2 = 4;  // T4

// FastLED arrays
#define NUM_LEDS 16
CRGB leds1[NUM_LEDS];
CRGB leds2[NUM_LEDS];

// States
bool fan_power = true;
int  fan_speed = 3;

bool switch_power = true;

bool light1_power = true;
int  light1_hue = 0;       // 0–360
int  light1_sat = 100;     // 0–100
int  light1_bri = 80;

bool light2_power = true;
int  light2_hue = 200;
int  light2_sat = 100;
int  light2_bri = 80;

// Devices
static Fan*             my_fan    = nullptr;
static LightBulb*       my_light1 = nullptr;
static LightBulb*       my_light2 = nullptr;
static Switch*          my_switch = nullptr;
static TemperatureSensor* my_temp = nullptr;

// Provisioning callback
void sysProvEvent(arduino_event_t *sys_event) {
    switch (sys_event->event_id) {
        case ARDUINO_EVENT_PROV_START:
#if CONFIG_IDF_TARGET_ESP32S2 || CONFIG_IDF_TARGET_ESP32S3
            Serial.printf("\nProvisioning SoftAP: %s | PoP: %s\n", service_name, pop);
            printQR(service_name, pop, "softap");
#else
            Serial.printf("\nProvisioning BLE: %s | PoP: %s\n", service_name, pop);
            printQR(service_name, pop, "ble");
#endif
            break;
        case ARDUINO_EVENT_PROV_INIT:
            wifi_prov_mgr_disable_auto_stop(10000);
            break;
        case ARDUINO_EVENT_PROV_CRED_SUCCESS:
            wifi_prov_mgr_stop_provisioning();
            break;
        default: break;
    }
}

// Update functions
void updateFan() {
    analogWrite(fan_pin, fan_power ? map(fan_speed, 1, 5, 80, 255) : 0);
}

void updateLight1() {
    if (light1_power) {
        CHSV hsv(light1_hue, map(light1_sat, 0, 100, 0, 255), map(light1_bri, 0, 100, 0, 255));
        leds1[0] = hsv;
        leds1[1] = hsv;
    } else {
        leds1[0] = CRGB::Black;
    }
    FastLED.show();
}

void updateLight2() {
    if (light2_power) {
        CHSV hsv(light2_hue, map(light2_sat, 0, 100, 0, 255), map(light2_bri, 0, 100, 0, 255));
        leds2[0] = hsv;
    } else {
        leds2[0] = CRGB::Black;
    }
    FastLED.show();
}

// RainMaker callback
void write_callback(Device *device, Param *param, const param_val_t val, void*, write_ctx_t*) {
    const char *dev = device->getDeviceName();
    const char *p   = param->getParamName();

    if (strcmp(dev, "Fan") == 0) {
        if (strcmp(p, ESP_RMAKER_DEF_POWER_NAME) == 0) { fan_power = val.val.b; updateFan(); }
        else if (strcmp(p, ESP_RMAKER_DEF_SPEED_NAME) == 0) { fan_speed = val.val.i; updateFan(); }
    }
    else if (strcmp(dev, "Light 1") == 0) {
        if (strcmp(p, ESP_RMAKER_DEF_POWER_NAME) == 0)       light1_power = val.val.b;
        else if (strcmp(p, ESP_RMAKER_DEF_HUE_NAME) == 0)        light1_hue = val.val.i;
        else if (strcmp(p, ESP_RMAKER_DEF_SATURATION_NAME) == 0) light1_sat = val.val.i;
        else if (strcmp(p, ESP_RMAKER_DEF_BRIGHTNESS_NAME) == 0) light1_bri = val.val.i;
        updateLight1();
    }
    else if (strcmp(dev, "Light 2") == 0) {
        if (strcmp(p, ESP_RMAKER_DEF_POWER_NAME) == 0)       light2_power = val.val.b;
        else if (strcmp(p, ESP_RMAKER_DEF_HUE_NAME) == 0)        light2_hue = val.val.i;
        else if (strcmp(p, ESP_RMAKER_DEF_SATURATION_NAME) == 0) light2_sat = val.val.i;
        else if (strcmp(p, ESP_RMAKER_DEF_BRIGHTNESS_NAME) == 0) light2_bri = val.val.i;
        updateLight2();
    }
    else if (strcmp(dev, "Switch") == 0) {
        if (strcmp(p, ESP_RMAKER_DEF_POWER_NAME) == 0) {
            switch_power = val.val.b;
            digitalWrite(switch_pin, switch_power);
        }
    }
    param->updateAndReport(val);
}

void setup() {
    Serial.begin(115200);

    pinMode(gpio_reset, INPUT);        // GPIO0 – no pullup needed (hardware has it)
    pinMode(fan_pin, OUTPUT);
    pinMode(switch_pin, OUTPUT);

    // FastLED – now compiles perfectly!
    FastLED.addLeds<WS2812B, LIGHT1_PIN, GRB>(leds1, NUM_LEDS);
    FastLED.addLeds<WS2812B, LIGHT2_PIN, GRB>(leds2, NUM_LEDS);
    FastLED.setBrightness(255);  // we control brightness in HSV

    updateLight1();
    updateLight2();
    updateFan();

    Node my_node = RMaker.initNode("IndusSmartHome");

    my_fan = new Fan("Fan");
    my_fan->addCb(write_callback);
    my_fan->addPowerParam(true);
    my_fan->addSpeedParam(3);

    my_light1 = new LightBulb("Light 1");
    my_light1->addCb(write_callback);
    my_light1->addPowerParam(true);
    my_light1->addBrightnessParam(80);
    my_light1->addHueParam(0);
    my_light1->addSaturationParam(100);

    my_light2 = new LightBulb("Light 2");
    my_light2->addCb(write_callback);
    my_light2->addPowerParam(true);
    my_light2->addBrightnessParam(80);
    my_light2->addHueParam(200);
    my_light2->addSaturationParam(100);

    my_switch = new Switch("Switch");
    my_switch->addCb(write_callback);
    my_switch->addPowerParam(true);

    my_temp = new TemperatureSensor("Chip Temp");

    my_node.addDevice(*my_fan);
    my_node.addDevice(*my_light1);
    my_node.addDevice(*my_light2);
    my_node.addDevice(*my_switch);
    my_node.addDevice(*my_temp);

    RMaker.enableOTA(OTA_USING_TOPICS);
    RMaker.enableTZService();
    RMaker.enableSchedule();
    RMaker.enableScenes();
    initAppInsights();

    RMaker.start();

    WiFi.onEvent(sysProvEvent);

#if CONFIG_IDF_TARGET_ESP32S2 || CONFIG_IDF_TARGET_ESP32S3
    WiFiProv.beginProvision(WIFI_PROV_SCHEME_SOFTAP, WIFI_PROV_SCHEME_HANDLER_NONE,
                            WIFI_PROV_SECURITY_1, pop, service_name);
#else
    WiFiProv.beginProvision(WIFI_PROV_SCHEME_BLE, WIFI_PROV_SCHEME_HANDLER_FREE_BTDM,
                            WIFI_PROV_SECURITY_1, pop, service_name);
#endif
}

void loop() {
    // EXACT ORIGINAL GPIO0 RESET LOGIC (from Espressif RainMaker examples)
    if (digitalRead(gpio_reset) == LOW) {
        delay(100); // debounce
        int startTime = millis();
        while (digitalRead(gpio_reset) == LOW) delay(50);
        int duration = millis() - startTime;

        if (duration > 10000) {
            Serial.println("Factory Reset");
            RMakerFactoryReset(2);
        } else if (duration > 3000) {
            Serial.println("Wi-Fi Reset");
            RMakerWiFiReset(2);
        } else {
            // Short press = toggle Switch
            switch_power = !switch_power;
            Serial.printf("Switch toggled to %s\n", switch_power ? "ON" : "OFF");
            my_switch->updateAndReportParam(ESP_RMAKER_DEF_POWER_NAME, switch_power);
            digitalWrite(switch_pin, switch_power);
        }
    }

    // Touch controls
    if (touchRead(touch_fan) < TOUCH_THRESHOLD) {
        fan_power = !fan_power;
        my_fan->updateAndReportParam(ESP_RMAKER_DEF_POWER_NAME, fan_power);
        updateFan();
        delay(300);
    }
    if (touchRead(touch_light1) < TOUCH_THRESHOLD) {
        light1_power = !light1_power;
        my_light1->updateAndReportParam(ESP_RMAKER_DEF_POWER_NAME, light1_power);
        updateLight1();
        delay(300);
    }
    if (touchRead(touch_light2) < TOUCH_THRESHOLD) {
        light2_power = !light2_power;
        my_light2->updateAndReportParam(ESP_RMAKER_DEF_POWER_NAME, light2_power);
        updateLight2();
        delay(300);
    }

    // Temperature every 10s
    static uint32_t lastTemp = 0;
    if (millis() - lastTemp > 10000) {
        float t = temperatureRead();
        my_temp->updateAndReportParam(ESP_RMAKER_DEF_TEMPERATURE_NAME, t);
        lastTemp = millis();
    }

    delay(100);
}