// ESP32-S2 TWAI (CAN) Vehicle IMU & Activity Sensor
// For Indusboard Coin or similar ESP32-S2 board
// Requires external CAN transceiver (SN65HVD230, TJA1050, etc.)
// LSM303AGR connected via I2C (default pins)

// Libraries
#include <Wire.h>
#include <LSM303AGR_ACC_Sensor.h>
#include <LSM303AGR_MAG_Sensor.h>
#include <driver/twai.h>          // ESP32 TWAI (CAN) driver

// Pin definitions - use GPIO_NUM_xx constants to avoid type errors
const gpio_num_t TWAI_TX_PIN = GPIO_NUM_40;   // Connect to transceiver TXD
const gpio_num_t TWAI_RX_PIN = GPIO_NUM_42;   // Connect to transceiver RXD

// I2C sensor objects
LSM303AGR_ACC_Sensor Acc(&Wire);
LSM303AGR_MAG_Sensor Mag(&Wire);

// Detection thresholds (tune these based on testing)
const float STATIC_THRESHOLD       = 0.12;     // g - below this → considered static
const float ACCELERATION_THRESHOLD = 0.25;     // g - positive X accel
const float BRAKE_THRESHOLD        = -0.30;    // g - negative X accel (braking)
const float COLLISION_THRESHOLD    = 2.5;      // g - sudden high acceleration
const float FREEFALL_THRESHOLD     = 0.25;     // g - very low total accel (fall/cliff)
const float ROLLOVER_ANGLE         = 65.0;     // degrees - tilt angle for rollover

// Speed estimation
float velocity_ms = 0.0f;           // m/s
float last_time = 0.0f;
unsigned long last_static_ms = 0;
const unsigned long STATIC_RESET_DELAY_MS = 2500;  // reset velocity after being static

// CAN message IDs (you can change these)
const uint32_t CAN_IMU_ID     = 0x100;   // Regular IMU data
const uint32_t CAN_EVENT_ID   = 0x200;   // Event + speed + status

void setup() {
  Serial.begin(115200);
  delay(100);
  Serial.println("\nESP32-S2 CAN IMU Vehicle Sensor Starting...");

  // Initialize I2C
  Wire.begin();

  // Initialize accelerometer
  if (Acc.begin() != 0 || Acc.Enable() != 0 || Acc.EnableTemperatureSensor() != 0) {
    Serial.println("ERROR: LSM303AGR Accelerometer init failed!");
    while (1) delay(100);
  }

  // Initialize magnetometer
  if (Mag.begin() != 0 || Mag.Enable() != 0) {
    Serial.println("ERROR: LSM303AGR Magnetometer init failed!");
    while (1) delay(100);
  }

  Serial.println("Sensors initialized");

  // TWAI / CAN configuration
  twai_general_config_t g_config = TWAI_GENERAL_CONFIG_DEFAULT(
      TWAI_TX_PIN,
      TWAI_RX_PIN,
      TWAI_MODE_NORMAL
  );

  twai_timing_config_t t_config = TWAI_TIMING_CONFIG_500KBITS();   // Common vehicle speed
  // Alternatives: TWAI_TIMING_CONFIG_250KBITS(), TWAI_TIMING_CONFIG_125KBITS()

  twai_filter_config_t f_config = TWAI_FILTER_CONFIG_ACCEPT_ALL();

  // Install TWAI driver
  if (twai_driver_install(&g_config, &t_config, &f_config) != ESP_OK) {
    Serial.println("ERROR: Failed to install TWAI driver");
    while (1) delay(100);
  }

  // Start TWAI
  if (twai_start() != ESP_OK) {
    Serial.println("ERROR: Failed to start TWAI");
    while (1) delay(100);
  }

  Serial.println("TWAI (CAN) initialized at 500 kbps");
  Serial.println("Ready - sending IMU & vehicle status via CAN");

  last_time = millis() / 1000.0f;
}

void loop() {
  // Read accelerometer (mg → g)
  int32_t accel_raw[3];
  Acc.GetAxes(accel_raw);
  float ax = accel_raw[0] / 1000.0f;
  float ay = accel_raw[1] / 1000.0f;
  float az = accel_raw[2] / 1000.0f;

  // Read temperature
  float temperature = 0.0f;
  Acc.GetTemperature(&temperature);

  // Read magnetometer (mGauss)
  int32_t mag_raw[3];
  Mag.GetAxes(mag_raw);

  // Total acceleration magnitude (approx, gravity removed on Z)
  float total_accel = sqrt(ax*ax + ay*ay + (az - 1.0f)*(az - 1.0f));

  // Time step for velocity integration
  float now = millis() / 1000.0f;
  float dt = now - last_time;
  last_time = now;

  // Integrate acceleration in X direction (forward/backward) to estimate speed
  velocity_ms += ax * 9.81f * dt;

  // Convert to km/h
  float velocity_kmh = velocity_ms * 3.6f;

  // Reset velocity if vehicle has been static for a while (prevents drift)
  if (total_accel < STATIC_THRESHOLD) {
    if (millis() - last_static_ms > STATIC_RESET_DELAY_MS) {
      velocity_ms = 0.0f;
      velocity_kmh = 0.0f;
    }
    last_static_ms = millis();
  } else {
    last_static_ms = millis();
  }

  // -----------------------------
  // Activity / Event Detection
  // -----------------------------
  String event = "";
  uint8_t event_code = 0;

  if (total_accel < STATIC_THRESHOLD) {
    event = "STATIC";
    event_code = 0;
  }
  else if (ax > ACCELERATION_THRESHOLD) {
    event = "ACCELERATION";
    event_code = 1;
  }
  else if (ax < BRAKE_THRESHOLD) {
    event = "BRAKE / DECELERATION";
    event_code = 2;
  }

  // Collision
  if (total_accel > COLLISION_THRESHOLD) {
    event = "COLLISION ACCIDENT";
    event_code = 3;
  }

  // Free fall / cliff fall
  float total_accel_raw = sqrt(ax*ax + ay*ay + az*az);
  if (total_accel_raw < FREEFALL_THRESHOLD) {
    event = "SUDDEN FALL / CLIFF";
    event_code = 4;
  }

  // Rollover detection (simple tilt)
  float roll_deg  = atan2(ay, az) * 180.0f / PI;
  float pitch_deg = atan2(-ax, sqrt(ay*ay + az*az)) * 180.0f / PI;
  if (abs(roll_deg) > ROLLOVER_ANGLE || abs(pitch_deg) > ROLLOVER_ANGLE) {
    event = "ROLLOVER ACCIDENT";
    event_code = 5;
  }

  // -----------------------------
  // Send CAN Messages
  // -----------------------------

  // 1. Regular IMU data (every loop)
  twai_message_t imu_msg;
  imu_msg.identifier = CAN_IMU_ID;
  imu_msg.extd = 0;                    // 11-bit standard ID
  imu_msg.rtr = 0;
  imu_msg.data_length_code = 8;

  int16_t imu_data[4];
  imu_data[0] = (int16_t)(ax * 1000);   // X accel (mg)
  imu_data[1] = (int16_t)(ay * 1000);   // Y accel
  imu_data[2] = (int16_t)(az * 1000);   // Z accel
  imu_data[3] = (int16_t)(temperature * 100);  // temp °C × 100

  memcpy(imu_msg.data, imu_data, 8);

  if (twai_transmit(&imu_msg, pdMS_TO_TICKS(10)) == ESP_OK) {
    // Success (you can comment this out)
  }

  // 2. Event message (only when event is detected)
  if (event_code != 0 || total_accel >= STATIC_THRESHOLD) {   // send on movement too
    twai_message_t event_msg;
    event_msg.identifier = CAN_EVENT_ID;
    event_msg.extd = 0;
    event_msg.rtr = 0;
    event_msg.data_length_code = 8;

    int16_t event_data[4];
    event_data[0] = (int16_t)(velocity_ms * 10);   // m/s × 10
    event_data[1] = (int16_t)(velocity_kmh * 10);  // km/h × 10
    event_data[2] = (int16_t)event_code;           // event type
    event_data[3] = (int16_t)(total_accel * 100);  // total accel g × 100

    memcpy(event_msg.data, event_data, 8);

    twai_transmit(&event_msg, pdMS_TO_TICKS(10));
  }

  // Serial debug output
  Serial.print("| Acc[g]: ");
  Serial.print(ax, 2); Serial.print(" ");
  Serial.print(ay, 2); Serial.print(" ");
  Serial.print(az, 2);

  Serial.print(" | Mag: ");
  Serial.print(mag_raw[0]); Serial.print(" ");
  Serial.print(mag_raw[1]); Serial.print(" ");
  Serial.print(mag_raw[2]);

  Serial.print(" | Temp: "); Serial.print(temperature, 1);

  Serial.print(" | Speed: "); Serial.print(velocity_kmh, 1); Serial.print(" km/h");

  if (event != "") {
    Serial.print(" | EVENT: "); Serial.print(event);
  }
  Serial.println();

  delay(80);  // ~12.5 Hz update rate - adjust as needed
}