// ESP32 TWAI (CAN) Receiver for Vehicle IMU & Activity Sensor
// Matches transmitter using IDs 0x100 (IMU) and 0x200 (Event/Speed)
// For Indusboard Coin / ESP32-S2 or any ESP32 with TWAI
// Requires external CAN transceiver (SN65HVD230, TJA1050, etc.)

#include <Arduino.h>
#include <driver/twai.h>

// Pin definitions - adjust to your board / wiring
const gpio_num_t TWAI_TX_PIN = GPIO_NUM_40;   // Connect to transceiver TXD
const gpio_num_t TWAI_RX_PIN = GPIO_NUM_42;   // Connect to transceiver RXD

// CAN message IDs (must match transmitter)
const uint32_t CAN_IMU_ID   = 0x100;
const uint32_t CAN_EVENT_ID = 0x200;

// Event codes (should match transmitter)
const char* event_names[] = {
  "STATIC",               // 0
  "ACCELERATION",         // 1
  "BRAKE / DECELERATION", // 2
  "COLLISION ACCIDENT",   // 3
  "SUDDEN FALL / CLIFF",  // 4
  "ROLLOVER ACCIDENT",    // 5
  "UNKNOWN"
};

void setup() {
  Serial.begin(115200);
  delay(200);
  Serial.println("\n=== ESP32 TWAI CAN Receiver (IMU + Events) ===");
  Serial.println("Waiting for messages from vehicle sensor...");

  // TWAI configuration
  twai_general_config_t g_config = TWAI_GENERAL_CONFIG_DEFAULT(
      TWAI_TX_PIN,
      TWAI_RX_PIN,
      TWAI_MODE_NORMAL
  );

  twai_timing_config_t t_config = TWAI_TIMING_CONFIG_500KBITS();

  // Accept all messages (you can later use filters if needed)
  twai_filter_config_t f_config = TWAI_FILTER_CONFIG_ACCEPT_ALL();

  // Install TWAI driver
  if (twai_driver_install(&g_config, &t_config, &f_config) != ESP_OK) {
    Serial.println("ERROR: Failed to install TWAI driver");
    while (1) delay(100);
  }

  // Start TWAI
  if (twai_start() != ESP_OK) {
    Serial.println("ERROR: Failed to start TWAI");
    while (1) delay(100);
  }

  Serial.print("TWAI (CAN) initialized at 500 kbps - RX on GPIO ");
  Serial.print(TWAI_RX_PIN);
  Serial.print(", TX on GPIO ");
  Serial.println(TWAI_TX_PIN);
  Serial.println("----------------------------------------");
}

void loop() {
  twai_message_t rx_msg;

  // Wait for a message (timeout 10 ms)
  if (twai_receive(&rx_msg, pdMS_TO_TICKS(10)) == ESP_OK) {

    // Optional: ignore RTR or extended frames if you want strict filtering
    if (rx_msg.rtr || rx_msg.extd) return;

    Serial.print("[CAN ID 0x");
    Serial.print(rx_msg.identifier, HEX);
    Serial.print("]  ");

    if (rx_msg.identifier == CAN_IMU_ID && rx_msg.data_length_code == 8) {
      // IMU message (ax, ay, az in mg, temp × 100)
      int16_t* data = (int16_t*)rx_msg.data;

      float ax = data[0] / 1000.0f;
      float ay = data[1] / 1000.0f;
      float az = data[2] / 1000.0f;
      float temp = data[3] / 100.0f;

      Serial.print("Accel: ");
      Serial.print(ax, 2); Serial.print(" ");
      Serial.print(ay, 2); Serial.print(" ");
      Serial.print(az, 2); Serial.print(" g");

      Serial.print("  | Temp: ");
      Serial.print(temp, 1);
      Serial.println(" °C");
    }

    else if (rx_msg.identifier == CAN_EVENT_ID && rx_msg.data_length_code == 8) {
      // Event + speed message
      int16_t* data = (int16_t*)rx_msg.data;

      float speed_ms  = data[0] / 10.0f;
      float speed_kmh = data[1] / 10.0f;
      uint16_t event_code = (uint16_t)data[2];   // unsigned interpretation
      float total_accel = data[3] / 100.0f;

      const char* event_str = (event_code < 6) ? event_names[event_code] : event_names[6];

      Serial.print("Speed: ");
      Serial.print(speed_kmh, 1);
      Serial.print(" km/h  (");
      Serial.print(speed_ms, 2);
      Serial.print(" m/s)");

      Serial.print("  | Event: ");
      Serial.print(event_str);

      Serial.print("  | Total accel: ");
      Serial.print(total_accel, 2);
      Serial.println(" g");
    }

    else {
      // Unknown message
      Serial.print("Data: ");
      for (int i = 0; i < rx_msg.data_length_code; i++) {
        if (rx_msg.data[i] < 0x10) Serial.print("0");
        Serial.print(rx_msg.data[i], HEX);
        Serial.print(" ");
      }
      Serial.println();
    }

    Serial.println("----------------------------------------");
  }
  // No message → short delay to prevent tight loop
  else {
    delay(5);
  }
}