// ESP32-S2 TWAI (CAN) Vehicle IMU & Activity Sensor with Web Dashboard (SSE Streaming)
// Station (STA) mode → connects to your existing WiFi router
// For Indusboard Coin or similar ESP32-S2 board
// Requires external CAN transceiver (SN65HVD230, TJA1050, etc.)
// LSM303AGR connected via I2C (default pins)
// Uses ESPAsyncWebServer + AsyncEventSource for SSE

#include <Wire.h>
#include <LSM303AGR_ACC_Sensor.h>
#include <LSM303AGR_MAG_Sensor.h>
#include <driver/twai.h>
#include <WiFi.h>
#include <ESPAsyncWebServer.h>
#include <AsyncEventSource.h>

// Pin definitions
const gpio_num_t TWAI_TX_PIN = GPIO_NUM_40;
const gpio_num_t TWAI_RX_PIN = GPIO_NUM_42;

// I2C sensor objects
LSM303AGR_ACC_Sensor Acc(&Wire);
LSM303AGR_MAG_Sensor Mag(&Wire);

// WiFi credentials - CHANGE THESE!
const char* ssid     = "2g";      // ← Replace
const char* password = "1234567890";  // ← Replace

AsyncWebServer server(80);
AsyncEventSource *events;

// HTML Dashboard (embedded)
const char* html_page PROGMEM = R"rawliteral(
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Vehicle IMU Dashboard</title>
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css" rel="stylesheet">
  <style>
    body { background: linear-gradient(135deg, #e0f7fa, #b2ebf2); font-family: 'Segoe UI', sans-serif; padding: 20px; }
    .container { max-width: 1200px; margin: auto; }
    .card { border-radius: 16px; box-shadow: 0 8px 25px rgba(0,0,0,0.12); transition: transform 0.3s; margin-bottom: 20px; }
    .card:hover { transform: translateY(-6px); }
    .car-card { background: linear-gradient(135deg, #ff6b6b, #ff9e2c); color: white; height: 320px; position: relative; overflow: hidden; }
    .imu-card { background: linear-gradient(135deg, #4facfe, #00f2fe); color: white; }
    .activity-card { background: linear-gradient(135deg, #a18cd1, #fbc2eb); color: white; }
    #car { font-size: 120px; position: absolute; left: 50%; top: 55%; transform: translate(-50%, -50%); transition: all 0.4s ease; }
    .alert-icon { font-size: 100px; position: absolute; left: 50%; top: 50%; transform: translate(-50%, -50%) scale(0); opacity: 0; transition: all 0.5s; color: #ff1744; text-shadow: 0 0 10px #000; }
    .active { transform: translate(-50%, -50%) scale(1) !important; opacity: 1 !important; }
    .value { font-size: 1.6rem; font-weight: bold; }
    #status { position: fixed; top: 20px; right: 20px; width: 28px; height: 28px; border-radius: 50%; background: #f44336; box-shadow: 0 0 15px #f44336; transition: all 0.4s; z-index: 1000; }
    .connected { background: #4caf50 !important; box-shadow: 0 0 15px #4caf50 !important; }
  </style>
</head>
<body>
  <div class="container">
    <h1 class="text-center my-4 fw-bold text-primary"><i class="fas fa-car-side me-2"></i>Vehicle IMU Dashboard</h1>

    <div class="row g-4">
      <div class="col-lg-6">
        <div class="card car-card text-center">
          <h3 class="mt-4">Vehicle Status</h3>
          <i class="fas fa-car" id="car"></i>
          <i class="fas fa-exclamation-triangle alert-icon" id="collision"></i>
          <i class="fas fa-sync alert-icon" id="rollover"></i>
          <i class="fas fa-arrow-down alert-icon" id="fall"></i>
        </div>
      </div>
      <div class="col-lg-6">
        <div class="card imu-card p-4">
          <h3 class="text-center mb-4">Real-Time IMU</h3>
          <p>Acc X: <span id="ax" class="value">--</span> g</p>
          <p>Acc Y: <span id="ay" class="value">--</span> g</p>
          <p>Acc Z: <span id="az" class="value">--</span> g</p>
          <p>Mag: <span id="mx" class="value">--</span> <span id="my" class="value">--</span> <span id="mz" class="value">--</span> mGauss</p>
          <p>Temp: <span id="temp" class="value">--</span> °C</p>
          <p>Speed: <span id="speed" class="value">--</span> km/h</p>
        </div>
      </div>
    </div>

    <div class="row g-4 mt-3">
      <div class="col-12">
        <div class="card graph-card p-4">
          <h4 class="text-center mb-4 text-danger">Acceleration Axes (g)</h4>
          <canvas id="accelChart" height="180"></canvas>
        </div>
      </div>
    </div>

    <div class="row g-4 mt-3">
      <div class="col-12">
        <div class="card activity-card text-center p-5">
          <h3>Current Activity</h3>
          <div id="activity" class="value mt-3">--</div>
        </div>
      </div>
    </div>

    <div class="text-center mt-5">
      <button id="logBtn" class="btn btn-success btn-lg shadow" disabled>
        <i class="fas fa-database me-2"></i>View & Download Log
      </button>
    </div>
  </div>

  <div id="status"></div>

  <!-- Log Modal -->
  <div id="logModal" class="modal fade" tabindex="-1">
    <div class="modal-dialog modal-xl modal-dialog-scrollable">
      <div class="modal-content">
        <div class="modal-header">
          <h5 class="modal-title">IMU Data Log</h5>
          <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
        </div>
        <div class="modal-body">
          <table class="table table-striped table-hover">
            <thead>
              <tr><th>Time</th><th>Acc X</th><th>Acc Y</th><th>Acc Z</th>
              <th>Mag X</th><th>Mag Y</th><th>Mag Z</th><th>Temp</th><th>Speed</th><th>Event</th></tr>
            </thead>
            <tbody id="logTable"></tbody>
          </table>
        </div>
        <div class="modal-footer">
          <button id="csvBtn" class="btn btn-info">Download CSV</button>
          <button id="txtBtn" class="btn btn-secondary">Download TXT</button>
          <button type="button" class="btn btn-danger" data-bs-dismiss="modal">Close</button>
        </div>
      </div>
    </div>
  </div>

  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
  <script>
    const statusDot = document.getElementById('status');
    const logBtn = document.getElementById('logBtn');
    const logModal = new bootstrap.Modal(document.getElementById('logModal'));
    const logTable = document.getElementById('logTable');
    const axEl = document.getElementById('ax');
    const ayEl = document.getElementById('ay');
    const azEl = document.getElementById('az');
    const mxEl = document.getElementById('mx');
    const myEl = document.getElementById('my');
    const mzEl = document.getElementById('mz');
    const tempEl = document.getElementById('temp');
    const speedEl = document.getElementById('speed');
    const activityEl = document.getElementById('activity');
    const car = document.getElementById('car');
    const collision = document.getElementById('collision');
    const rollover = document.getElementById('rollover');
    const fall = document.getElementById('fall');

    let logData = [];

    const accelChart = new Chart(document.getElementById('accelChart'), {
      type: 'line',
      data: { labels: [], datasets: [
        {label:'X', data:[], borderColor:'red', tension:0.3},
        {label:'Y', data:[], borderColor:'green', tension:0.3},
        {label:'Z', data:[], borderColor:'blue', tension:0.3}
      ]},
      options: {scales:{y:{beginAtZero:false}}, animation:false, plugins:{legend:{position:'top'}}}
    });

    const es = new EventSource('/stream');

    es.onopen = () => {
      console.log('SSE connected');
      statusDot.classList.add('connected');
      logBtn.disabled = false;
    };

    es.onerror = () => {
      console.log('SSE error');
      statusDot.classList.remove('connected');
      logBtn.disabled = true;
    };

    es.onmessage = (e) => {
      try {
        const parts = e.data.split(',');
        if (parts.length >= 12) {
          const d = {
            ax: parseFloat(parts[0]),
            ay: parseFloat(parts[1]),
            az: parseFloat(parts[2]),
            mx: parseInt(parts[3]),
            my: parseInt(parts[4]),
            mz: parseInt(parts[5]),
            temp: parseFloat(parts[6]),
            speed: parseFloat(parts[7]),
            eventCode: parseInt(parts[8]),
            roll: parseFloat(parts[9]),
            yaw: parseFloat(parts[10]),
            pitch: parseFloat(parts[11])
          };

          axEl.textContent = isNaN(d.ax) ? '--' : d.ax.toFixed(3);
          ayEl.textContent = isNaN(d.ay) ? '--' : d.ay.toFixed(3);
          azEl.textContent = isNaN(d.az) ? '--' : d.az.toFixed(3);
          mxEl.textContent = isNaN(d.mx) ? '--' : d.mx;
          myEl.textContent = isNaN(d.my) ? '--' : d.my;
          mzEl.textContent = isNaN(d.mz) ? '--' : d.mz;
          tempEl.textContent = isNaN(d.temp) ? '--' : d.temp.toFixed(1);
          speedEl.textContent = isNaN(d.speed) ? '--' : d.speed.toFixed(1);

          let act = 'STATIC';
          if (d.eventCode === 1) act = 'ACCELERATION';
          else if (d.eventCode === 2) act = 'BRAKE / DECEL';
          else if (d.eventCode === 3) act = 'COLLISION';
          else if (d.eventCode === 4) act = 'FALL / CLIFF';
          else if (d.eventCode === 5) act = 'ROLLOVER';
          activityEl.textContent = act;

          // Car animation - always fa-car icon, only transform changes
          let offset = (d.speed || 0) * 3;
          let heading = d.yaw || 0;
          let rollTilt  = (d.roll  || 0) * 1.5;   // stronger for visibility
          let pitchTilt = (d.pitch || 0) * 1.0;

          car.style.transform = 
            `translate(-50%, -50%) ` +
            `translateX(${offset}px) ` +
            `rotateZ(${heading}deg) ` +
            `rotateX(${pitchTilt}deg) ` +
            `rotateY(${rollTilt}deg)`;

          // Make rollover/tilt more dramatic
          if (Math.abs(d.roll) > 45 || Math.abs(d.pitch) > 45) {
            car.style.transition = 'all 0.25s ease-out';
            car.style.filter = 'drop-shadow(0 0 15px #ff1744)';
          } else {
            car.style.transition = 'all 0.4s ease';
            car.style.filter = 'none';
          }

          // Alert icons (overlay - do NOT replace car icon)
          collision.classList.toggle('active', d.eventCode === 3);
          rollover.classList.toggle('active', d.eventCode === 5);
          fall.classList.toggle('active', d.eventCode === 4);

          // Chart
          const t = new Date().toLocaleTimeString([], {minute:'2-digit', second:'2-digit'});
          accelChart.data.labels.push(t);
          accelChart.data.datasets[0].data.push(d.ax);
          accelChart.data.datasets[1].data.push(d.ay);
          accelChart.data.datasets[2].data.push(d.az);
          if (accelChart.data.labels.length > 60) {
            accelChart.data.labels.shift();
            accelChart.data.datasets.forEach(ds => ds.data.shift());
          }
          accelChart.update('none');

          // Log
          logData.push({
            time: t,
            ax: isNaN(d.ax)?'--':d.ax.toFixed(3),
            ay: isNaN(d.ay)?'--':d.ay.toFixed(3),
            az: isNaN(d.az)?'--':d.az.toFixed(3),
            mx: isNaN(d.mx)?'--':d.mx,
            my: isNaN(d.my)?'--':d.my,
            mz: isNaN(d.mz)?'--':d.mz,
            temp: isNaN(d.temp)?'--':d.temp.toFixed(1),
            speed: isNaN(d.speed)?'--':d.speed.toFixed(1),
            event: act
          });
        }
      } catch(e) {
        console.error('Parse error:', e);
      }
    };

    logBtn.onclick = () => {
      logTable.innerHTML = '';
      logData.forEach(item => {
        logTable.innerHTML += `<tr><td>${item.time}</td><td>${item.ax}</td><td>${item.ay}</td><td>${item.az}</td><td>${item.mx}</td><td>${item.my}</td><td>${item.mz}</td><td>${item.temp}</td><td>${item.speed}</td><td>${item.event}</td></tr>`;
      });
      logModal.show();
    };

    document.getElementById('csvBtn').onclick = () => {
      let csv = 'Time,AccX,AccY,AccZ,MagX,MagY,MagZ,Temp,Speed,Event\n';
      logData.forEach(r => csv += `${r.time},${r.ax},${r.ay},${r.az},${r.mx},${r.my},${r.mz},${r.temp},${r.speed},${r.event}\n`);
      const blob = new Blob([csv], {type: 'text/csv'});
      const url = URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url; a.download = 'imu_log.csv'; a.click(); URL.revokeObjectURL(url);
    };

    document.getElementById('txtBtn').onclick = () => {
      let txt = 'Vehicle IMU Log\n\n';
      logData.forEach(r => txt += `${r.time} | Acc: ${r.ax} ${r.ay} ${r.az} | Mag: ${r.mx} ${r.my} ${r.mz} | Temp: ${r.temp} | Speed: ${r.speed} km/h | ${r.event}\n`);
      const blob = new Blob([txt], {type: 'text/plain'});
      const url = URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url; a.download = 'imu_log.txt'; a.click(); URL.revokeObjectURL(url);
    };
  </script>
</body>
</html>
)rawliteral";

// Thresholds & globals
const float STATIC_THRESHOLD       = 0.12f;
const float ACCELERATION_THRESHOLD = 0.25f;
const float BRAKE_THRESHOLD        = -0.30f;
const float COLLISION_THRESHOLD    = 2.5f;
const float FREEFALL_THRESHOLD     = 0.25f;
const float ROLLOVER_ANGLE         = 65.0f;

float velocity_ms = 0.0f;
float last_time = 0.0f;
unsigned long last_static_ms = 0;
const unsigned long STATIC_RESET_DELAY_MS = 2500;

const uint32_t CAN_IMU_ID   = 0x100;
const uint32_t CAN_EVENT_ID = 0x200;

void setup() {
  Serial.begin(115200);
  delay(100);
  Serial.println("\nESP32-S2 CAN IMU Vehicle Sensor (STA mode)...");

  Wire.begin();

  if (Acc.begin() != 0 || Acc.Enable() != 0 || Acc.EnableTemperatureSensor() != 0) {
    Serial.println("Acc init failed!");
    while(1) delay(100);
  }

  if (Mag.begin() != 0 || Mag.Enable() != 0) {
    Serial.println("Mag init failed!");
    while(1) delay(100);
  }

  twai_general_config_t g_config = TWAI_GENERAL_CONFIG_DEFAULT(TWAI_TX_PIN, TWAI_RX_PIN, TWAI_MODE_NORMAL);
  twai_timing_config_t t_config = TWAI_TIMING_CONFIG_500KBITS();
  twai_filter_config_t f_config = TWAI_FILTER_CONFIG_ACCEPT_ALL();

  if (twai_driver_install(&g_config, &t_config, &f_config) != ESP_OK ||
      twai_start() != ESP_OK) {
    Serial.println("TWAI init failed!");
    while(1) delay(100);
  }

  WiFi.mode(WIFI_STA);
  WiFi.begin(ssid, password);
  Serial.print("Connecting WiFi ");
  while (WiFi.status() != WL_CONNECTED) {
    delay(500); Serial.print(".");
  }
  Serial.println("\nConnected! IP: " + WiFi.localIP().toString());

  server.on("/", HTTP_GET, [](AsyncWebServerRequest *request){
    request->send_P(200, "text/html", html_page);
  });

  events = new AsyncEventSource("/stream");
  server.addHandler(events);
  server.begin();

  Serial.println("Server started - open IP in browser");
  last_time = millis() / 1000.0f;
}

void loop() {
  int32_t accel_raw[3];
  Acc.GetAxes(accel_raw);
  float ax = accel_raw[0] / 1000.0f;
  float ay = accel_raw[1] / 1000.0f;
  float az = accel_raw[2] / 1000.0f;

  float temperature = 0.0f;
  Acc.GetTemperature(&temperature);

  int32_t mag_raw[3];
  Mag.GetAxes(mag_raw);

  float yaw_deg   = atan2(mag_raw[1], mag_raw[0]) * 180.0f / PI;
  float roll_deg  = atan2(ay, az) * 180.0f / PI;
  float pitch_deg = atan2(-ax, sqrt(ay*ay + az*az)) * 180.0f / PI;

  float total_accel = sqrt(ax*ax + ay*ay + (az - 1.0f)*(az - 1.0f));

  float now = millis() / 1000.0f;
  float dt = now - last_time;
  last_time = now;

  velocity_ms += ax * 9.81f * dt;
  float velocity_kmh = velocity_ms * 3.6f;

  if (total_accel < STATIC_THRESHOLD) {
    if (millis() - last_static_ms > STATIC_RESET_DELAY_MS) {
      velocity_ms = velocity_kmh = 0.0f;
    }
    last_static_ms = millis();
  } else {
    last_static_ms = millis();
  }

  uint8_t event_code = 0;
  if (total_accel < STATIC_THRESHOLD) event_code = 0;
  else if (ax > ACCELERATION_THRESHOLD) event_code = 1;
  else if (ax < BRAKE_THRESHOLD) event_code = 2;

  if (total_accel > COLLISION_THRESHOLD) event_code = 3;

  float total_accel_raw = sqrt(ax*ax + ay*ay + az*az);
  if (total_accel_raw < FREEFALL_THRESHOLD) event_code = 4;

  if (abs(roll_deg) > ROLLOVER_ANGLE || abs(pitch_deg) > ROLLOVER_ANGLE) event_code = 5;

  // CAN IMU message - safe initialization
  twai_message_t imu_msg = {0};
  imu_msg.identifier       = CAN_IMU_ID;
  imu_msg.extd             = 0;
  imu_msg.rtr              = 0;
  imu_msg.data_length_code = 8;
  int16_t imu_data[4] = {
      (int16_t)(ax * 1000),
      (int16_t)(ay * 1000),
      (int16_t)(az * 1000),
      (int16_t)(temperature * 100)
  };
  memcpy(imu_msg.data, imu_data, 8);
  twai_transmit(&imu_msg, pdMS_TO_TICKS(10));

  // CAN Event message - safe initialization
  if (event_code != 0 || total_accel >= STATIC_THRESHOLD) {
    twai_message_t event_msg = {0};
    event_msg.identifier       = CAN_EVENT_ID;
    event_msg.extd             = 0;
    event_msg.rtr              = 0;
    event_msg.data_length_code = 8;
    int16_t event_data[4] = {
        (int16_t)(velocity_ms * 10),
        (int16_t)(velocity_kmh * 10),
        (int16_t)event_code,
        (int16_t)(total_accel * 100)
    };
    memcpy(event_msg.data, event_data, 8);
    twai_transmit(&event_msg, pdMS_TO_TICKS(10));
  }

  // SSE CSV (12 fields)
  String dataStr = String(ax,3) + "," + String(ay,3) + "," + String(az,3) + "," +
                   String(mag_raw[0]) + "," + String(mag_raw[1]) + "," + String(mag_raw[2]) + "," +
                   String(temperature,1) + "," + String(velocity_kmh,1) + "," +
                   String(event_code) + "," + String(roll_deg,1) + "," +
                   String(yaw_deg,1) + "," + String(pitch_deg,1);

  if (events) events->send(dataStr.c_str(), NULL, millis());

  // Debug output
  Serial.printf("Acc:%.2f %.2f %.2f | Mag:%d %d %d | T:%.1f | Sp:%.1f | Ev:%d | R:%.1f P:%.1f Y:%.1f\n",
                ax,ay,az, mag_raw[0],mag_raw[1],mag_raw[2], temperature, velocity_kmh,
                event_code, roll_deg, pitch_deg, yaw_deg);

  delay(80);
}