from maix import camera, display, image, app, time, gpio, pinmap, touchscreen
import os
# to make videography from jpeg images
# ssh into the MaixCam board. Go to /snapshot and run the following command
# ffmpeg -framerate 10 -pattern_type glob -i 'snapshot_*.jpg' -c:v mjpeg timelapse.avi
# Encoder 'libx264' is not available on Maixcam.
# Run this code on your laptop to convert the video to *.mp4 if required.
# ffmpeg -i timelapse.avi -vcodec libx264 -crf 23 timelapse.mp4. 

cam = camera.Camera()
disp = display.Display()
ts = touchscreen.TouchScreen()

snapshot_dir = "/snapshot"
if not os.path.exists(snapshot_dir):
    os.mkdir(snapshot_dir)

# Initial values
lapse_time_list = [30, 150,  450, 600, 900]          # 0.5m, 2.5m, 7.5m, 10m, 15m
max_duration_list = [900, 1800, 2700, 7200, 14400]     # 15m, 30m, 45m, 2h, 4h
lapse_index = 0
duration_index = 3

lapse_time = lapse_time_list[lapse_index]
max_duration = max_duration_list[duration_index]

last_snapshot_time = 0
snap_count = 0
start_time = time.time()

BTN_W = 120
BTN_H = 40

# UI positions
back_btn_pos = (390, 10, BTN_W, BTN_H)  # Adjusted for better touch accuracy
lapse_cycle_btn = (10, 10, BTN_W, BTN_H)
duration_cycle_btn = (190, 10, BTN_W, BTN_H)

# Static labels with fixed positions
lapse_buttons = [(f"{v/60:.1f}m", v, (10, 60 + i * 50, BTN_W, BTN_H)) for i, v in enumerate(lapse_time_list)]
duration_buttons = [(f"{v/60}m", v, (190, 60 + i * 50, BTN_W, BTN_H)) for i, v in enumerate(max_duration_list)]

def draw_rectangle(img, x, y, w, h, color=image.COLOR_RED, thickness=2):
    img.draw_line(x, y, x + w, y, color=color, thickness=thickness)
    img.draw_line(x, y, x, y + h, color=color, thickness=thickness)
    img.draw_line(x + w, y, x + w, y + h, color=color, thickness=thickness)
    img.draw_line(x, y + h, x + w, y + h, color=color, thickness=thickness)

def draw_value_rects(img, buttons, selected_idx):
    for i, (label, _, (x, y, w, h)) in enumerate(buttons):
        color = image.COLOR_GREEN if i == selected_idx else image.COLOR_RED
        draw_rectangle(img, x, y, w, h, color=color, thickness=2)
        img.draw_string(x + 10, y + 10, label, scale=2, color=color)

def draw_cycle_buttons(img):
    # Lapse NEXT
    lx, ly, lw, lh = lapse_cycle_btn
    draw_rectangle(img, lx, ly, lw, lh, color=image.COLOR_BLUE)
    img.draw_string(lx + 20, ly + 10, "NEXT", scale=2, color=image.COLOR_BLUE)

    # Duration NEXT
    dx, dy, dw, dh = duration_cycle_btn
    draw_rectangle(img, dx, dy, dw, dh, color=image.COLOR_BLUE)
    img.draw_string(dx + 20, dy + 10, "NEXT", scale=2, color=image.COLOR_BLUE)

def in_box(x, y, box):
    bx, by, bw, bh = box
    return bx <= x <= bx + bw and by <= y <= by + bh

# Touch debounce state
just_touched = False

while True:
    current_time = time.time()
    elapsed_time = current_time - start_time

    if elapsed_time >= max_duration:
        print(f"Session complete after {max_duration} seconds. Total snapshots: {snap_count}")
        break

    img = cam.read()
    img_for_display = img.copy()  # keep a clean version for saving

    time_left = max(0, max_duration - elapsed_time)
    minutes_left = time_left // 60

    img_for_display.draw_string(10, 450, f"Lap:{lapse_time/60:0.1f}m Max:{max_duration//60}m Lft:{minutes_left}m:{snap_count}", color=image.COLOR_YELLOW, scale=2)

    # BACK button
    bx, by, bw, bh = back_btn_pos
    draw_rectangle(img_for_display, bx, by, bw, bh, color=image.COLOR_RED, thickness=2)
    img_for_display.draw_string(bx + 20, by + 10, "BACK", scale=2, color=image.COLOR_RED)

    draw_value_rects(img_for_display, lapse_buttons, lapse_index)
    draw_value_rects(img_for_display, duration_buttons, duration_index)
    draw_cycle_buttons(img_for_display)

    #disp.show(img)
    disp.show(img_for_display)
    tx, ty, pressed = ts.read()

    if pressed and not just_touched:
        just_touched = True
        print(f"Touch at ({tx}, {ty})")

        if tx > 0 and ty > 0:
            if in_box(tx, ty, back_btn_pos):
                print("Back pressed. Exiting.")
                break
            elif in_box(tx, ty, lapse_cycle_btn):
                lapse_index = (lapse_index + 1) % len(lapse_time_list)
                lapse_time = lapse_time_list[lapse_index]
                print(f"Lapse changed to {lapse_time // 60} min")
            elif in_box(tx, ty, duration_cycle_btn):
                duration_index = (duration_index + 1) % len(max_duration_list)
                max_duration = max_duration_list[duration_index]
                print(f"Duration changed to {max_duration // 60} min")

    elif not pressed and just_touched:
        time.sleep_ms(100)
        just_touched = False

    # Save snapshot
    if current_time - last_snapshot_time >= lapse_time:
        formatted_time = time.localtime()
        time_str = f"{formatted_time.year}-{formatted_time.month:02d}-{formatted_time.day:02d}_{formatted_time.hour:02d}-{formatted_time.minute:02d}-{formatted_time.second:02d}"
        img.draw_string(5, 5, time_str, color=image.COLOR_WHITE, scale=1)
        save_path = f"{snapshot_dir}/snapshot_{time_str}.jpg"
        img.save(save_path)
        snap_count += 1
        print(f"[{snap_count}] Snapshot saved: {save_path}")
        last_snapshot_time = current_time